package ru.yandex.direct.jobs.yt.audit;

import java.util.ArrayDeque;
import java.util.Collection;

import com.google.common.base.Preconditions;

/**
 * Очередь с ограничением на максимальное количество элементов.
 * Если очередной добавляемый элемент не помещается в неё, то он "выталкивает" элемент
 * с противоположного конца очереди.
 *
 * @param <E>
 */
public class SizeLimitedQueue<E> extends ArrayDeque<E> {
    private final int sizeLimit;

    public SizeLimitedQueue(int sizeLimit) {
        Preconditions.checkArgument(sizeLimit > 0);
        this.sizeLimit = sizeLimit;
    }

    @Override
    public void addFirst(E e) {
        if (size() == sizeLimit) {
            removeLast();
        }
        super.addFirst(e);
    }

    @Override
    public void addLast(E e) {
        if (size() == sizeLimit) {
            removeFirst();
        }
        super.addLast(e);
    }

    @Override
    public boolean addAll(Collection<? extends E> collection) {
        for (E item : collection) {
            add(item);
        }
        return !collection.isEmpty();
    }

    @Override
    public boolean offerFirst(E e) {
        if (size() == sizeLimit) {
            removeLast();
        }
        return super.offerFirst(e);
    }

    @Override
    public boolean offerLast(E e) {
        if (size() == sizeLimit) {
            removeFirst();
        }
        return super.offerLast(e);
    }

    @Override
    public boolean add(E e) {
        if (size() == sizeLimit) {
            removeFirst();
        }
        return super.add(e);
    }

    @Override
    public boolean offer(E e) {
        if (size() == sizeLimit) {
            removeFirst();
        }
        return super.offer(e);
    }

    @Override
    public void push(E e) {
        if (size() == sizeLimit) {
            removeLast();
        }
        super.push(e);
    }
}
