package ru.yandex.direct.jobs.yt.audit;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.ytchecksum.YtChecksum;
import ru.yandex.direct.core.entity.ytchecksum.YtChecksumMethod;
import ru.yandex.direct.core.entity.ytchecksum.YtChecksumStatus;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapper;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.mysql.schema.ColumnSchema;
import ru.yandex.direct.mysql.schema.KeyColumn;
import ru.yandex.direct.mysql.schema.KeySchema;
import ru.yandex.direct.mysql.schema.TableSchema;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

public class TableChecker {
    private final Logger logger = LoggerFactory.getLogger(TableChecker.class);

    private final DatabaseWrapperProvider databaseWrapperProvider;

    private final TableCheckParams params;
    private final YtChecksumYtRepository ytRepository;
    private final String dbName;
    private final int desiredChunkSize;

    // Первая колонка в первичном ключе может быть одним из этих типов
    // строковые типы пока не поддерживаем, экзотику типа mediumblob тоже
    private static final Set<String> SUPPORTED_FIRST_PK_COLUMN_TYPES = ImmutableSet.of(
            "tinyint", "smallint", "mediumint", "int", "bigint"
    );

    // Какие колонки умеем сравнивать
    private static final Set<String> SUPPORTED_CONTENT_COLUMN_TYPES = ImmutableSet.of(
            "tinyint", "smallint", "mediumint", "int", "bigint",
            "decimal", "numeric",
            "float", "double",
            "bit",
            "date", "datetime", "time", "timestamp",
            "char", "varchar", "text", "enum", "set",
            "json"
    );

    // Типы данных, порядок на которых зависит от выбранного collation
    private static final Set<String> STRING_TYPES = ImmutableSet.of(
            "char", "varchar", "text"
    );

    public TableChecker(DatabaseWrapperProvider databaseWrapperProvider,
                        TableCheckParams params,
                        YtChecksumYtRepository ytRepository) {
        this.databaseWrapperProvider = databaseWrapperProvider;
        this.params = params;
        this.ytRepository = ytRepository;
        this.dbName = params.getDb();
        this.desiredChunkSize = params.getDesiredChunkSize();
    }

    private TableSchema loadTableSchema(String table) throws SQLException {
        DatabaseWrapper databaseWrapper = databaseWrapperProvider.get(dbName);
        try (Connection connection = databaseWrapper.getDataSource().getConnection()) {
            return TableSchema.dump(connection, table);
        }
    }

    @Nullable
    private ColumnSchema firstPKColumn(TableSchema tableSchema) {
        KeySchema primaryKey = tableSchema.getKeys().stream().filter(KeySchema::isPrimary).findFirst().orElse(null);
        return primaryKey != null ? tableSchema.findColumn(primaryKey.getColumns().get(0).getName()) : null;
    }

    private boolean canProcessTable(TableSchema tableSchema) {
        ColumnSchema firstColumn = firstPKColumn(tableSchema);
        if (firstColumn == null) {
            logger.error("Skip table {}: missing primary key", tableSchema.getName());
            return false;
        }
        if (!SUPPORTED_FIRST_PK_COLUMN_TYPES.contains(firstColumn.getDataType())) {
            logger.error("Skip table {}: unsupported PK type {}", tableSchema.getName(), firstColumn.getDataType());
            return false;
        }
        return true;
    }

    private long explainRowsCount(TableSchema tableSchema, BigInteger min, BigInteger max) throws SQLException {
        String key = firstPKColumn(tableSchema).getName();
        String where = String.format("(%s >= %d and %s <= %d)", key, min, key, max);
        String sql = String.format("explain select * from %s where %s", tableSchema.getName(), where);
        DatabaseWrapper databaseWrapper = databaseWrapperProvider.get(dbName);
        try (Connection connection = databaseWrapper.getDataSource().getConnection()) {
            try (PreparedStatement statement = connection.prepareStatement(sql)) {
                try (ResultSet resultSet = statement.executeQuery()) {
                    resultSet.next();
                    return resultSet.getLong("rows");
                }
            }
        }
    }

    private long selectRowsCount(TableSchema tableSchema, BigInteger min, BigInteger maxExclude) throws SQLException {
        String key = firstPKColumn(tableSchema).getName();
        String where = String.format("%s >= %d and %s <= %d", key, min, key, maxExclude.subtract(BigInteger.ONE));
        String sql = String.format("select count(*) as cnt from %s where (%s)", tableSchema.getName(), where);
        DatabaseWrapper databaseWrapper = databaseWrapperProvider.get(dbName);
        try (Connection connection = databaseWrapper.getDataSource().getConnection()) {
            try (PreparedStatement statement = connection.prepareStatement(sql)) {
                try (ResultSet resultSet = statement.executeQuery()) {
                    resultSet.next();
                    return resultSet.getLong("cnt");
                }
            }
        }
    }

    public TableCheckResult processTable() {
        try {
            return processTableCore();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    private TableCheckResult processTableCore() throws SQLException {
        DatabaseWrapper databaseWrapper = databaseWrapperProvider.get(dbName);

        // Сначала удалим старые записи
        if (params.getIteration() == -1) {
            // Специальные ручные запуски работают с iteration = -1
            logger.info("Deleting all records with iteration=-1 from yt_checksum");
            try (Connection connection = databaseWrapper.getDataSource().getConnection()) {
                try (PreparedStatement statement = connection.prepareStatement(
                        "delete from yt_checksum where iteration = -1")) {
                    int updated = statement.executeUpdate();
                    logger.info("Deleted {} rows", updated);
                }
            }
        }
        if (params.isDeleteOld()) {
            Instant threshold = Instant.now().minus(14, ChronoUnit.DAYS);
            logger.info("Deleting old content from yt_checksum (older than {})", threshold);
            try (Connection connection = databaseWrapper.getDataSource().getConnection()) {
                try (PreparedStatement statement = connection.prepareStatement(
                        "delete from yt_checksum where ts < ?"
                )) {
                    statement.setTimestamp(1, new Timestamp(threshold.toEpochMilli()));
                    int updated = statement.executeUpdate();
                    logger.info("Deleted {} rows", updated);
                }
            }
        }

        // Получить схему таблицы, проверить, умеем ли мы обрабатывать такие таблицы
        TableSchema tableSchema = loadTableSchema(params.getTable());
        if (!canProcessTable(tableSchema)) {
            return new TableCheckResult(true, emptyList());
        }

        // Поделим таблицу на чанки
        logger.info("Estimating table chunks..");
        List<Chunk> chunks = getTableChunks(tableSchema, databaseWrapper);

        List<NumberedChunk> numberedChunks = new ArrayList<>();
        if (chunks.isEmpty()) {
            // Для пустой таблицы создаём чанк, проверяющий всё её содержимое
            NumberedChunk wholeTableChunk = new NumberedChunk(0, -1, new Chunk(null, null, 0));
            numberedChunks.add(wholeTableChunk);
        } else {
            for (int i = 0; i < chunks.size(); i++) {
                numberedChunks.add(new NumberedChunk(i, null, chunks.get(i)));
            }
        }
        TreeMap<Integer, NumberedChunk> chunkMap = numberedChunks.stream()
                .collect(toMap(NumberedChunk::getNumber, Function.identity(), (a, b) -> {
                    throw new IllegalStateException("Duplicate key");
                }, TreeMap::new));

        // Разобьём их на пачки и поинсертим в mysql
        List<List<NumberedChunk>> batches = makeBatches(numberedChunks, params.getBatchSize());
        List<YtChecksum> allMismatched = new ArrayList<>();
        int failedCount = 0;
        for (List<NumberedChunk> batchChunks : batches) {
            List<NumberedChunk> chunksToSend = getChunksToSend(batchChunks, params);
            List<YtChecksum> mismatched = new ArrayList<>();
            while (!chunksToSend.isEmpty()) {
                for (var chunk : chunksToSend) {
                    // В основном наборе чанков не используем ToStr
                    sendChunk(chunk, tableSchema, false);
                }
                var replicated = waitForBatchReplicated(chunksToSend, Duration.ofSeconds(10), Duration.ofMinutes(45));
                // Если контрольная сумма на стороне YT не вычисляется, то нет смысла продолжать
                if (replicated.stream().allMatch(
                        c -> c.getStatus() == YtChecksumStatus.STARTED && c.getYtCrc() == null)) {
                    logger.info("Seems replica doesn't verify checksums, exiting");
                    markFinished(databaseWrapper);
                    return new TableCheckResult(true, emptyList());
                }
                mismatched.addAll(replicated.stream()
                        .filter(c -> c.getStatus() == YtChecksumStatus.STARTED)
                        .filter(c -> !Objects.equals(c.getThisCrc(), c.getYtCrc()) || !Objects.equals(c.getThisStr(),
                                c.getYtStr()))
                        .collect(toList()));
                List<YtChecksum> failed = replicated.stream()
                        .filter(c -> c.getStatus() != YtChecksumStatus.STARTED)
                        .collect(toList());

                int failedChunksLimit = params.getFailedChunksLimit();
                failedCount += failed.size();
                logger.info("Replicated {} chunks. {} failed. Total failed count {} (max {})",
                        replicated.size(), failed.size(), failedCount, failedChunksLimit);
                if (failedCount > failedChunksLimit) {
                    logger.error("Failed chunks limit exceeded: {} (max {})", failedCount, failedChunksLimit);
                    throw new RuntimeException("Failed chunks limit exceeded");
                }
                if (!failed.isEmpty()) {
                    List<NumberedChunk> splittedChunks = new ArrayList<>();
                    for (var failedChecksum : failed) {
                        NumberedChunk failedChunk = chunkMap.get(failedChecksum.getChunk());
                        if (!failedChunk.canBeSplitted()) {
                            // Тут уже ничего сделать нельзя: возможно, чанк слишком велик
                            // даже для одного значения первой колонки PK
                            // В будущем можно доработать так, чтобы использовались все колонки ключа
                            logger.error("Failed chunk can't be splitted: {}", failedCount);
                            throw new RuntimeException("Failed chunk can't be splitted");
                        }
                        splittedChunks.addAll(splitChunk(failedChunk, chunkMap, tableSchema));
                    }
                    chunksToSend = splittedChunks;
                } else {
                    chunksToSend = emptyList();
                }
            }
            allMismatched.addAll(mismatched);
            int mismatchCount = mismatched.size();
            logger.info("Batch was replicated, {} with checksum mismatch", mismatchCount);
            if (mismatchCount != 0 && params.isStopAfterFirstMismatch()) {
                logger.info("Found mismatches, stopping generating new chunks");
                break;
            }
        }
        logger.info("All batches were replicated. Total {} mismatches found", allMismatched.size());

        // Теперь будем искать конкретные строки с расхождениями
        int limit = params.getFailedRowsLimit();
        List<YtChecksum> singleRows = new ArrayList<>();
        if (!allMismatched.isEmpty()) {
            logger.info("Starting search for failed rows");
            SizeLimitedQueue<YtChecksum> mismatches = new SizeLimitedQueue<>(limit);
            for (int i = 0; i < Math.min(allMismatched.size(), limit); i++) {
                mismatches.addLast(allMismatched.get(i));
            }
            //
            while (singleRows.size() < limit && !mismatches.isEmpty()) {
                YtChecksum checksum = mismatches.pollFirst();
                NumberedChunk chunk = chunkMap.get(checksum.getChunk());
                if (!chunk.canBeSplitted() && checksum.getMethod() == YtChecksumMethod.TOSTRING) {
                    logger.info("Found #{} mismatch: {}, diff\n{}\n{}", singleRows.size(), chunk,
                            checksum.getThisStr(), checksum.getYtStr());
                    singleRows.add(checksum);
                    continue;
                }
                //
                List<NumberedChunk> toSend;
                if (chunk.canBeSplitted()) {
                    toSend = splitChunk(chunk, chunkMap, tableSchema);
                } else {
                    // Этот чанк уже нельзя разбивать, но он получен с помощью другого метода сравнения
                    // Нужно его переотправить как есть, но уже с помощью метода ToString
                    toSend = singletonList(repeatChunk(chunk, chunkMap));
                }
                for (var chunkToSend : toSend) {
                    sendChunk(chunkToSend, tableSchema, true);
                }
                //
                var replicated = waitForBatchReplicated(toSend, Duration.ofSeconds(3), Duration.ofMinutes(5));
                for (YtChecksum ytChecksum : replicated) {
                    if (!Objects.equals(ytChecksum.getThisCrc(), ytChecksum.getYtCrc())
                            || !Objects.equals(ytChecksum.getThisStr(), ytChecksum.getYtStr())) {
                        logger.info("Mismatched chunk: {}", chunkMap.get(ytChecksum.getChunk()));
                        mismatches.addFirst(ytChecksum);
                    } else {
                        logger.info("OK chunk: {}", chunkMap.get(ytChecksum.getChunk()));
                    }
                }
            }

            logger.info("Finished failed rows search. Found {} failed rows.", singleRows.size());
            for (int i = 0; i < singleRows.size(); i++) {
                YtChecksum singleRow = singleRows.get(i);
                logger.info("Found #{} mismatch: {}, diff\n{}\n{}", i,
                        chunkMap.get(singleRow.getChunk()), singleRow.getThisStr(), singleRow.getYtStr());
            }
        }

        // Если схема таблицы не изменилась во время проверки, то считаем операцию успешно
        // завершённой и устанавливаем на чанках статус Finished
        TableSchema tableSchemaAfter = loadTableSchema(params.getTable());
        if (tableSchema.schemaEquals(tableSchemaAfter)) {
            markFinished(databaseWrapper);
        } else {
            throw new RuntimeException(String.format("Schema of table %s has changed", params.getTable()));
        }

        logger.info("Finished checking table {}", params.getTable());
        return new TableCheckResult(true, singleRows);
    }

    private void markFinished(DatabaseWrapper databaseWrapper) throws SQLException {
        try (Connection connection = databaseWrapper.getDataSource().getConnection()) {
            try (PreparedStatement statement = connection.prepareStatement(
                    "update yt_checksum set status = 'Finished' where tbl = ? and iteration = ? and status = 'Started'"
            )) {
                statement.setString(1, makeTbl(params.getTable(), params.getDestYtCluster()));
                statement.setInt(2, params.getIteration());
                int rows = statement.executeUpdate();
                logger.info("Iteration marked as finished, updated {} rows", rows);
            }
        }
    }

    private List<NumberedChunk> splitChunk(NumberedChunk chunk,
                                           TreeMap<Integer, NumberedChunk> chunkMap,
                                           TableSchema tableSchema)
            throws SQLException {
        Preconditions.checkArgument(chunk.canBeSplitted());
        int maxNumber = chunkMap.isEmpty() ? 0 : chunkMap.lastKey();
        BigInteger mid = chunk.getMin().add(chunk.getMax()).divide(BigInteger.TWO);
        NumberedChunk left = new NumberedChunk(maxNumber + 1, chunk.getNumber(),
                new Chunk(chunk.getMin(), mid, selectRowsCount(tableSchema, chunk.getMin(), mid)));
        NumberedChunk right = new NumberedChunk(maxNumber + 2, chunk.getNumber(),
                new Chunk(mid, chunk.getMax(), selectRowsCount(tableSchema, mid, chunk.getMax())));
        chunkMap.put(left.getNumber(), left);
        chunkMap.put(right.getNumber(), right);
        logger.info("Split chunk {} to {} and {}", chunk, left, right);
        return Arrays.asList(left, right);
    }

    private NumberedChunk repeatChunk(NumberedChunk chunk,
                                      TreeMap<Integer, NumberedChunk> chunkMap) {
        Preconditions.checkArgument(!chunk.canBeSplitted());
        int maxNumber = chunkMap.isEmpty() ? 0 : chunkMap.lastKey();
        NumberedChunk repeated = new NumberedChunk(maxNumber + 1, chunk.getNumber(), chunk.getChunk());
        chunkMap.put(repeated.getNumber(), repeated);
        logger.info("Repeat chunk {} as {}", chunk, repeated);
        return repeated;
    }

    @Nullable
    private Pair<BigInteger, BigInteger> getMinMaxGlobal(TableSchema tableSchema,
                                                         DatabaseWrapper databaseWrapper) throws SQLException {
        String keyColumn = firstPKColumn(tableSchema).getName();
        BigInteger min, max;
        try (Connection connection = databaseWrapper.getDataSource().getConnection()) {
            try (PreparedStatement statement = connection.prepareStatement(String.format("" +
                            "select min(%s) as min_value, max(%s) as max_value from %s",
                    keyColumn, keyColumn, tableSchema.getName()))) {
                try (ResultSet resultSet = statement.executeQuery()) {
                    resultSet.next();
                    min = Optional.ofNullable(resultSet.getBigDecimal("min_value"))
                            .map(BigDecimal::toBigIntegerExact)
                            .orElse(null);
                    max = Optional.ofNullable(resultSet.getBigDecimal("max_value"))
                            .map(BigDecimal::toBigIntegerExact)
                            .orElse(null);
                }
            }
        }
        return (min == null || max == null) ? null : Pair.of(min, max);
    }

    private BigInteger max(BigInteger a, BigInteger b) {
        return a.max(b);
    }

    private BigInteger min(BigInteger a, BigInteger b) {
        return a.min(b);
    }

    /**
     * Получает min и max значения ключей для таблицы с целью проверить приблизительно
     * percent процентов содержимого (по количеству строк)
     * Работает не очень точно, т.к. explain в mysql не очень качественно оценивает кардинальность
     */
    private Pair<BigInteger, BigInteger> getMinMax(TableSchema tableSchema,
                                                   DatabaseWrapper databaseWrapper,
                                                   int percent) throws SQLException {
        Preconditions.checkArgument(percent > 0 && percent < 100);

        Pair<BigInteger, BigInteger> minMax = getMinMaxGlobal(tableSchema, databaseWrapper);
        if (minMax == null) {
            return null;
        }

        BigInteger globalMin = minMax.getLeft();
        BigInteger globalMax = minMax.getRight();

        // Нужно определить, сколько примерно строк нам нужно
        // Здесь используется целочисленная арифметика: точность не так важна
        long totalCnt = explainRowsCount(tableSchema, globalMin, globalMax);
        long needCnt = totalCnt * percent / 100;

        // Примерно определяем, где может быть граница нужного нам отрезка
        BigInteger min = globalMax.subtract(
                globalMax.subtract(globalMin).multiply(BigInteger.valueOf(percent)).divide(BigInteger.valueOf(100))
        );
        min = max(min, globalMin);
        min = min(min, globalMax);

        // Теперь у нас есть интервал [min; globalMax]
        // В нём сколько-то строк. А нам нужно примерно needCnt строк
        // поэтому мы будем расширять или сужать его, пока не достигнем желаемого количества,
        // или пока не упрёмся в globalMin/globalMax, или не достигнем лимита операций
        long cnt = explainRowsCount(tableSchema, min, globalMax);
        BigInteger fixedDelta = max(BigInteger.ONE, globalMax.subtract(min).divide(BigInteger.valueOf(20)));
        while (cnt > 2 * needCnt && min.compareTo(globalMax) < 0) {
            min = min(min.add(max(BigInteger.ONE, globalMax.subtract(min).divide(BigInteger.valueOf(10)))), globalMax);
            cnt = explainRowsCount(tableSchema, min, globalMax);
        }
        int iterationsCount = 0;
        while (cnt * 2 < needCnt && min.compareTo(globalMin) > 0 && iterationsCount < 500) {
            // Наращиваем интервал более плавно (аддитивно), т.к. explain
            // имеет свойство быстро и внезапно увеличиваться на порядки при шевелении диапазона
            min = max(min.subtract(fixedDelta), globalMin);
            cnt = explainRowsCount(tableSchema, min, globalMax);
            iterationsCount++;
        }

        return Pair.of(min, globalMax);
    }

    private List<Chunk> getTableChunks(TableSchema tableSchema,
                                       DatabaseWrapper databaseWrapper) throws SQLException {
        Pair<BigInteger, BigInteger> globalMinMax = getMinMaxGlobal(tableSchema, databaseWrapper);
        if (globalMinMax == null) {
            logger.info("Table {} is empty", tableSchema.getName());
            return emptyList();
        }
        long globalCnt = explainRowsCount(tableSchema, globalMinMax.getLeft(), globalMinMax.getRight());

        if (isKeyCollationDependent(tableSchema) && globalCnt > 5_000_000) {
            logger.error("Can't process table {}: PK is collation-dependent and table has many rows: {}",
                    tableSchema.getName(), globalCnt);
            throw new RuntimeException("Can't process table " + tableSchema.getName());
        }

        Pair<BigInteger, BigInteger> minMax;
        if (globalCnt > 50_000_000) {
            // Для самых больших таблиц проверяем только 10% содержимого
            logger.info("Table is huge, 10% of data will be checked");
            minMax = getMinMax(tableSchema, databaseWrapper, 10);
        } else if (globalCnt > 10_000_000) {
            // Таблицы поменьше проверяем на 25%
            logger.info("Table is large, 25% of data will be checked");
            minMax = getMinMax(tableSchema, databaseWrapper, 25);
        } else {
            // Обычные таблицы проверяем целиком
            logger.info("Table is normal, 100% of data will be checked");
            minMax = globalMinMax;
        }
        if (minMax == null) {
            logger.info("Table {} is empty", tableSchema.getName());
            return emptyList();
        }

        BigInteger min = minMax.getLeft();
        BigInteger max = minMax.getRight();

        long totalCnt = explainRowsCount(tableSchema, min, max);
        long chunksCnt = totalCnt / desiredChunkSize + (totalCnt % desiredChunkSize != 0 ? 1 : 0);
        BigInteger idsPerChunk = max.subtract(min).add(BigInteger.ONE).divide(BigInteger.valueOf(chunksCnt));

        ArrayDeque<Chunk> chunks = new ArrayDeque<>();
        BigInteger minId = min;
        while (minId.compareTo(max) <= 0) {
            BigInteger maxId = min(max.add(BigInteger.ONE), minId.add(idsPerChunk));
            chunks.add(new Chunk(minId, maxId, selectRowsCount(tableSchema, minId, maxId)));
            minId = maxId;
        }
        logger.info("Got {} initial chunks", chunks.size());

        List<Chunk> finalChunks = new ArrayList<>();
        while (!chunks.isEmpty()) {
            Chunk chunk = chunks.poll();
            // Слишком большие чанки разбиваем на более мелкие (если это возможно)
            if (chunk.cnt > 2 * desiredChunkSize && chunk.max.subtract(chunk.min).compareTo(BigInteger.ONE) > 0) {
                BigInteger mid = chunk.min.add(chunk.max).divide(BigInteger.TWO);
                Chunk first = new Chunk(chunk.min, mid, selectRowsCount(tableSchema, chunk.min, mid));
                Chunk second = new Chunk(mid, chunk.max, selectRowsCount(tableSchema, mid, chunk.max));
                chunks.addFirst(second);
                chunks.addFirst(first);
                continue;
            }
            // А слишком мелкие объединяем
            Chunk nextChunk = chunks.peek();
            if (nextChunk != null && chunk.cnt + nextChunk.cnt < desiredChunkSize) {
                chunks.remove();
                chunks.addFirst(new Chunk(chunk.min, nextChunk.max, chunk.cnt + nextChunk.cnt));
                continue;
            }
            // Чанки подходящего размера уходят в финальный список
            finalChunks.add(chunk);
        }
        logger.info("Found {} chunks", finalChunks.size());
        for (int i = 0; i < finalChunks.size(); i++) {
            Chunk chunk = finalChunks.get(i);
            logger.info("Chunk #{}: {}-{}, cnt={}", i, chunk.min, chunk.max, chunk.cnt);
        }
        logger.info("Rows to be checked: {}", finalChunks.stream().map(Chunk::getCnt).reduce(Long::sum).orElse(0L));
        return finalChunks;
    }

    private List<YtChecksum> waitForBatchReplicated(Collection<NumberedChunk> sentChunks,
                                                    Duration delay,
                                                    Duration timeout) {
        logger.info("Starting waiting {} chunks to be replicated into YT", sentChunks.size());
        LocalDateTime start = LocalDateTime.now();
        LocalDateTime deadline = start.plus(timeout);
        List<YtChecksum> replicated = emptyList();
        while (replicated.size() < sentChunks.size()) {
            if (LocalDateTime.now().isAfter(deadline)) {
                logger.info("Waiting timed out (deadline was {})", deadline);
                throw new RuntimeException("Waiting timed out");
            }
            try {
                logger.info("Waiting {} sec before next try", delay.toSeconds());
                Thread.sleep(delay.toMillis());
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("Thread was interrupted", e);
            }
            replicated = ytRepository.getReplicated(sentChunks, params);
            logger.info("Got {} replicated chunks", replicated.size());
        }
        return replicated;
    }

    private void sendChunk(NumberedChunk chunk, TableSchema tableSchema, boolean useToStr) throws SQLException {
        Preconditions.checkState(chunk.getMin() == null || chunk.getMax() == null
                || chunk.getMin().compareTo(chunk.getMax()) < 0);

        List<ColumnSchema> columnsToUse = getColumnsToUse(tableSchema, new HashSet<>(params.getExcludeColumns()));
        String columnsUsed = getColumnsUsedMask(tableSchema, columnsToUse);
        String rowContentSql = getRowContentSql(columnsToUse);

        // Отдельные строки можем сравнивать через ToString напрямую
        // но может оказаться, что одному значению первой колонки первичного ключа соответствует
        // несколько строк. В таком случае ToString возьмёт только первые 10К символов
        YtChecksumMethod method = useToStr && !chunk.canBeSplitted()
                ? YtChecksumMethod.TOSTRING
                : YtChecksumMethod.CRC;
        String chunkSql = getChunkSql(tableSchema, rowContentSql, columnsUsed, params.getIteration(), chunk, method);
        logger.info("Inserting chunk #{}={} using SQL {}", chunk.getNumber(), chunk, chunkSql);

        insertChunk(chunkSql);
    }

    private List<NumberedChunk> getChunksToSend(List<NumberedChunk> chunks, TableCheckParams params) {
        List<NumberedChunk> result = new ArrayList<>();
        for (NumberedChunk chunk : chunks) {
            int chunkNum = chunk.getNumber();
            if (params.getMinChunkNumber() != null && chunkNum < params.getMinChunkNumber()) {
                logger.info("Skip chunk #{} because < minChunkNumber={}", chunkNum, params.getMinChunkNumber());
                continue;
            }
            if (params.getMaxChunkNumber() != null && chunkNum > params.getMaxChunkNumber()) {
                logger.info("Skip chunk #{} because > maxChunkNumber={}", chunkNum, params.getMaxChunkNumber());
                continue;
            }
            result.add(chunk);
        }
        return result;
    }

    private List<List<NumberedChunk>> makeBatches(List<NumberedChunk> chunks, int batchSize) {
        List<List<NumberedChunk>> batches = new ArrayList<>();
        for (int startBatch = 0; startBatch < chunks.size(); startBatch += batchSize) {
            List<NumberedChunk> batchChunks = new ArrayList<>();
            for (int offset = 0; offset < batchSize && startBatch + offset < chunks.size(); offset++) {
                batchChunks.add(chunks.get(startBatch + offset));
            }
            batches.add(batchChunks);
        }
        return batches;
    }

    private static final int TO_STR_MAX_LEN = 1024 * 10;

    private void insertChunk(String chunkSql) throws SQLException {
        DatabaseWrapper databaseWrapper = databaseWrapperProvider.get(dbName);
        try (Connection connection = databaseWrapper.getDataSource().getConnection()) {
            int isolationLevel = connection.getTransactionIsolation();
            connection.setTransactionIsolation(Connection.TRANSACTION_REPEATABLE_READ);
            try (Statement statement = connection.createStatement()) {
                // Ставим лимит в 10К символов для GROUP_CONCAT (нужно для method='ToStr')
                // Умножаем на 4, т.к. переменная group_concat_max_len ограничивает кол-во в байтах,
                // а UTF-8 символы могут занимать до 4 байт
                statement.addBatch("SET group_concat_max_len = " + TO_STR_MAX_LEN * 4);
                // Инициализация переменных для подсчёта контрольной суммы (нужно для method='Crc')
                statement.addBatch("SET @cnt = 0, @crc = ''");
                statement.addBatch(chunkSql);
                statement.executeBatch();
            } finally {
                connection.setTransactionIsolation(isolationLevel);
            }
        }
    }

    private List<ColumnSchema> getColumnsToUse(TableSchema tableSchema, Set<String> excludeColumns) {
        return tableSchema.getColumns().stream()
                // Пропускаем колонки с неподдерживаемыми типами
                // Сейчас к примеру под фильтрацию попадают mediumblob, year, float, double
                // Таких колонок в базе всего несколько, и пока нет смысла заморачиваться,
                // тем более эти типы мы не планируем дальше широко использовать в mysql
                .filter(columnSchema -> SUPPORTED_CONTENT_COLUMN_TYPES.contains(columnSchema.getDataType()))
                // Пропускаем колонки из blacklist
                .filter(columnSchema -> !excludeColumns.contains(columnSchema.getName()))
                .collect(toList());
    }

    private String getColumnsUsedMask(TableSchema tableSchema, List<ColumnSchema> columnsToUse) {
        Set<String> used = columnsToUse.stream().map(ColumnSchema::getName).collect(toSet());
        return tableSchema.getColumns().stream()
                .map(ColumnSchema::getName)
                .map(column -> used.contains(column) ? "1" : "0")
                .collect(Collectors.joining());
    }

    private String getRowContentSql(List<ColumnSchema> columnsToUse) {
        String contentColumns = columnsToUse.stream()
                .map(columnSchema -> {
                    String column = "`" + columnSchema.getName() + "`";
                    String type = columnSchema.getDataType();
                    switch (type) {
                        case "double":
                        case "float":
                            // В оригинальной percona-checksum использовался ROUND(),
                            // но выяснилось, что округление в mysql с особенностями
                            // https://st.yandex-team.ru/DIRECT-111498#5e820a28ba955701d5e2bc82
                            // поэтому вместо ROUND(value, 2) используем TRUNCATE(value, 3)
                            // Дополнительно делаем каст к DECIMAL, чтобы не терять точность при TRUNCATE
                            // Иначе можно получить странное -- например, mysql вычисляет
                            // truncate(1615e-2, 3) = 16.149
                            return "TRUNCATE(CAST(" + column + "AS DECIMAL(65, 7)), 3)";
                        case "timestamp":
                        case "datetime":
                            return "CASE WHEN " + column + " = '0000-00-00 00:00:00' THEN NULL ELSE " + column + " END";
                        case "date":
                            return "CASE WHEN " + column + " = '0000-00-00' THEN NULL ELSE " + column + " END";
                        case "time":
                            return "CASE WHEN " + column + " = '00:00:00' THEN NULL ELSE " + column + " END";
                        default:
                            return column;
                    }
                })
                .collect(Collectors.joining(", "));
        String isNullColumns = columnsToUse.stream()
                .filter(ColumnSchema::isNullable)
                .map(ColumnSchema::getName)
                .map(column -> "ISNULL(`" + column + "`)")
                .collect(Collectors.joining(", "));
        String isNullColumnsSql = isNullColumns.isEmpty() ? "''" : String.format("CONCAT(%s)", isNullColumns);
        String contentColumnsSql = contentColumns.isEmpty() ? "''" : contentColumns;
        return String.format("CONCAT_WS('#', %s, %s)", contentColumnsSql, isNullColumnsSql);
    }

    private String getBoundaries(String key, NumberedChunk chunk) {
        String escapedKey = "`" + key + "`";
        String left = chunk.getMin() != null
                ? String.format("%s >= %d", escapedKey, chunk.getMin())
                : "1=1";
        String right = chunk.getMax() != null
                ? String.format("%s <= %d", escapedKey, chunk.getMax().subtract(BigInteger.ONE))
                : "1=1";
        return String.format("(%s and %s)", left, right);
    }

    private boolean isKeyCollationDependent(TableSchema tableSchema) {
        KeySchema keySchema = tableSchema.getPrimaryKey().get();
        return keySchema.getColumns().stream()
                .anyMatch(col -> STRING_TYPES.contains(tableSchema.findColumn(col.getName()).getDataType()));
    }

    private String getOrderByKeySql(TableSchema tableSchema) {
        KeySchema keySchema = tableSchema.getPrimaryKey().get();
        return keySchema.getColumns().stream()
                .map(KeyColumn::getName)
                .map(s -> "`" + s + "`")
                .collect(Collectors.joining(", "));
    }

    private String getChunkSql(TableSchema tableSchema, String rowContentSql, String columnsUsed,
                               int iteration, NumberedChunk chunk, YtChecksumMethod method) {
        if (isKeyCollationDependent(tableSchema)) {
            return getChunkSqlWithCollate(tableSchema, rowContentSql, columnsUsed, iteration, chunk, method);
        } else {
            return getChunkSqlOrderByPK(tableSchema, rowContentSql, columnsUsed, iteration, chunk, method);
        }
    }

    private String getChunkSqlOrderByPK(TableSchema tableSchema, String rowContentSql, String columnsUsed,
                                        int iteration, NumberedChunk chunk, YtChecksumMethod method) {
        String tableName = makeTbl(tableSchema.getName(), params.getDestYtCluster());
        String key = firstPKColumn(tableSchema).getName();
        String boundaries = getBoundaries(key, chunk);
        if (method == YtChecksumMethod.CRC) {
            return String.format("" +
                            " INSERT INTO yt_checksum (tbl, iteration, chunk, parent_chunk, boundaries, method," +
                            " this_crc, this_crc_cnt, this_str, this_nulls_cnt, yt_crc, yt_crc_cnt, yt_str," +
                            " yt_nulls_cnt, columns_used)" +
                            " SELECT '%s', %d, %d, %d, '%s', 'Crc'," +
                            " COALESCE(RIGHT(MAX(" +
                            "   @crc := CONCAT(LPAD(@cnt := @cnt + 1, 16, '0'), CONV(CAST(CRC32(CONCAT(@crc, CRC32(" +
                            "     %s" +
                            "   ))) AS UNSIGNED), 10, 16))" +
                            " ), 16), 0) AS crc," +
                            " COUNT(*) AS cnt," +
                            " null, null, null, null, null, null, '%s'" +
                            " FROM %s FORCE INDEX (`PRIMARY`) WHERE %s;",
                    tableName,
                    iteration,
                    chunk.getNumber(),
                    ObjectUtils.firstNonNull(chunk.getParent(), -1),
                    boundaries,
                    rowContentSql,
                    columnsUsed,
                    tableSchema.getName(),
                    boundaries
            );
        } else if (method == YtChecksumMethod.TOSTRING) {
            // Используем INSERT IGNORE, т.к. если GROUP_CONCAT() обрежет данные, то jdbc драйвер
            // превратит этот варнинг в исключение, а строчка не вставится
            // И простых способов этого избежать не нашлось
            return String.format("" +
                            " INSERT IGNORE INTO yt_checksum (tbl, iteration, chunk, parent_chunk, boundaries," +
                            " method, this_crc, this_crc_cnt, this_str, this_nulls_cnt, yt_crc, yt_crc_cnt, yt_str," +
                            " yt_nulls_cnt, columns_used)" +
                            " SELECT '%s', %d, %d, %d, '%s', 'ToString', null, null," +
                            " LEFT(GROUP_CONCAT(%s SEPARATOR '\\n\\n'), %d), null, null, null, null, null, '%s'" +
                            " FROM %s FORCE INDEX (`PRIMARY`) WHERE %s;",
                    tableName,
                    iteration,
                    chunk.getNumber(),
                    ObjectUtils.firstNonNull(chunk.getParent(), -1),
                    boundaries,
                    rowContentSql,
                    TO_STR_MAX_LEN,
                    columnsUsed,
                    tableSchema.getName(),
                    boundaries
            );
        } else {
            throw new UnsupportedOperationException();
        }
    }

    private String getChunkSqlWithCollate(TableSchema tableSchema, String rowContentSql, String columnsUsed,
                                          int iteration, NumberedChunk chunk, YtChecksumMethod method) {
        String tableName = makeTbl(tableSchema.getName(), params.getDestYtCluster());
        String key = firstPKColumn(tableSchema).getName();
        String boundaries = getBoundaries(key, chunk);
        if (method == YtChecksumMethod.CRC) {
            return String.format("" +
                            " INSERT INTO yt_checksum (tbl, iteration, chunk, parent_chunk, boundaries, method," +
                            " this_crc, this_crc_cnt, this_str, this_nulls_cnt, yt_crc, yt_crc_cnt, yt_str," +
                            " yt_nulls_cnt, columns_used)" +
                            " SELECT '%s', %d, %d, %d, '%s', 'Crc'," +
                            " COALESCE(RIGHT(MAX(t.crc), 16), 0) AS crc," +
                            " COUNT(*) AS cnt," +
                            " null, null, null, null, null, null, '%s'" +
                            " FROM (" +
                            "   SELECT @crc := CONCAT(LPAD(@cnt:=@cnt+1,16,'0'), CONV(CAST(CRC32(CONCAT(@crc, CRC32(" +
                            "     %s" +
                            "   ))) AS UNSIGNED), 10, 16)) AS crc" +
                            "   FROM %s WHERE %s ORDER BY %s COLLATE utf8_bin" +
                            " ) AS t;",
                    tableName,
                    iteration,
                    chunk.getNumber(),
                    ObjectUtils.firstNonNull(chunk.getParent(), -1),
                    boundaries,
                    columnsUsed,
                    rowContentSql,
                    tableSchema.getName(),
                    boundaries,
                    getOrderByKeySql(tableSchema)
            );
        } else if (method == YtChecksumMethod.TOSTRING) {
            // Используем INSERT IGNORE, т.к. если GROUP_CONCAT() обрежет данные, то jdbc драйвер
            // превратит этот варнинг в исключение, а строчка не вставится
            // И простых способов этого избежать не нашлось
            return String.format("" +
                            " INSERT IGNORE INTO yt_checksum (tbl, iteration, chunk, parent_chunk, boundaries," +
                            " method, this_crc, this_crc_cnt, this_str, this_nulls_cnt, yt_crc, yt_crc_cnt, yt_str," +
                            " yt_nulls_cnt, columns_used)" +
                            " SELECT '%s', %d, %d, %d, '%s', 'ToString', null, null," +
                            " LEFT(GROUP_CONCAT(%s ORDER BY %s COLLATE utf8_bin SEPARATOR '\\n\\n'), %d)," +
                            " null, null, null, null, null, '%s'" +
                            " FROM %s WHERE %s;",
                    tableName,
                    iteration,
                    chunk.getNumber(),
                    ObjectUtils.firstNonNull(chunk.getParent(), -1),
                    boundaries,
                    rowContentSql,
                    getOrderByKeySql(tableSchema),
                    TO_STR_MAX_LEN,
                    columnsUsed,
                    tableSchema.getName(),
                    boundaries
            );
        } else {
            throw new UnsupportedOperationException();
        }
    }

    public static String parseTable(String tbl) {
        if (!tbl.contains("/")) {
            return tbl;
        }
        return tbl.split("/")[0];
    }

    public static String makeTbl(String table, YtCluster ytCluster) {
        return table + "/" + ytCluster.getName();
    }
}
