package ru.yandex.direct.autobudget.restart.repository

import com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule
import com.fasterxml.jackson.module.kotlin.readValue
import com.fasterxml.jackson.module.kotlin.registerKotlinModule
import org.jooq.DSLContext
import org.jooq.impl.DSL
import org.jooq.util.mysql.MySQLDSL
import org.springframework.stereotype.Component
import ru.yandex.direct.autobudget.restart.model.StrategyDto
import ru.yandex.direct.autobudget.restart.service.StrategyState
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapperhelper.InsertHelper
import ru.yandex.direct.utils.JsonUtils
import java.time.LocalDateTime

data class RestartTimes(
    val restartTime: LocalDateTime,
    val softRestartTime: LocalDateTime,
    val restartReason: String
)

data class CampRestartData(
    val cid: Long,
    val orderId: Long? = 0,
    val strategyData: StrategyDto,
    val times: RestartTimes,
    val state: StrategyState
)

enum class SaveMode {
    UPDATE_EXISTING,
    IGNORE_EXISTING,
    FAIL_EXISTING,
    UPDATE_EXISTING_STATE,
}

@Component
open class CampaignAutobudgetRestartRepository(
    private val dslProvider: DslContextProvider
) {
    private val restartTbl = Tables.CAMP_AUTOBUDGET_RESTART
    private val campaigns = Tables.CAMPAIGNS
    private val mapper = ObjectMapper()
        .registerKotlinModule()
        .registerModule(JavaTimeModule())
        .registerModule(JsonUtils.createBigDecimalModule())
        .disable(FAIL_ON_UNKNOWN_PROPERTIES)

    fun getAutobudgetRestartData(shard: Int, campaignIds: List<Long>): List<CampRestartData> {
        return getAutobudgetRestartData(dslProvider.ppc(shard), campaignIds, false)
    }

    fun getAutobudgetRestartData(
        dslContext: DSLContext,
        campaignIds: List<Long>,
        withUpdateLock: Boolean
    ): List<CampRestartData> {
        val selectStep = dslContext.select(
            restartTbl.CID,
            restartTbl.RESTART_TIME, restartTbl.SOFT_RESTART_TIME, restartTbl.RESTART_REASON,
            restartTbl.STRATEGY_DATA,
            restartTbl.STOP_TIME,
            campaigns.ORDER_ID
        )
            .from(restartTbl).join(campaigns).on(Tables.CAMPAIGNS.CID.eq(Tables.CAMP_AUTOBUDGET_RESTART.CID))
            .where(restartTbl.CID.`in`(campaignIds))

        val resultStep = if (withUpdateLock) selectStep.forUpdate() else selectStep

        return resultStep
            .fetch()
            .map {
                CampRestartData(
                    cid = it.getValue(restartTbl.CID),
                    orderId = it.getValue(campaigns.ORDER_ID),
                    times = RestartTimes(
                        restartTime = it.getValue(restartTbl.RESTART_TIME),
                        softRestartTime = it.getValue(restartTbl.SOFT_RESTART_TIME),
                        restartReason = it.getValue(restartTbl.RESTART_REASON),
                    ),
                    strategyData = mapper.readValue(it.getValue(restartTbl.STRATEGY_DATA)),
                    state = StrategyState(it.getValue(restartTbl.STOP_TIME))
                )
            }
    }

    fun saveAutobudgetRestartData(shard: Int, data: List<CampRestartData>, mode: SaveMode = SaveMode.UPDATE_EXISTING) {
        saveAutobudgetRestartData(dslProvider.ppc(shard), data, mode)
    }

    fun saveAutobudgetRestartData(
        dslContext: DSLContext,
        data: List<CampRestartData>,
        mode: SaveMode = SaveMode.UPDATE_EXISTING
    ) {
        if (data.isEmpty())
            return
        val insertHelper = InsertHelper(dslContext, restartTbl)
        data.forEach {
            insertHelper.set(restartTbl.CID, it.cid)
                .set(restartTbl.RESTART_TIME, it.times.restartTime)
                .set(restartTbl.SOFT_RESTART_TIME, it.times.softRestartTime)
                .set(restartTbl.RESTART_REASON, it.times.restartReason)
                .set(restartTbl.STRATEGY_DATA, mapper.writeValueAsString(it.strategyData))
                .set(restartTbl.STOP_TIME, it.state.stopTime)
                .newRecord()
        }

        when (mode) {
            SaveMode.UPDATE_EXISTING ->
                insertHelper.onDuplicateKeyUpdate()
                    .set(restartTbl.RESTART_TIME, MySQLDSL.values(restartTbl.RESTART_TIME))
                    .set(restartTbl.SOFT_RESTART_TIME, MySQLDSL.values(restartTbl.SOFT_RESTART_TIME))
                    .set(restartTbl.RESTART_REASON, MySQLDSL.values(restartTbl.RESTART_REASON))
                    .set(restartTbl.STRATEGY_DATA, MySQLDSL.values(restartTbl.STRATEGY_DATA))
                    .set(
                        restartTbl.STOP_TIME,
                        DSL.greatest(
                            DSL.nvl(restartTbl.STOP_TIME, MySQLDSL.values(restartTbl.STOP_TIME)),
                            MySQLDSL.values(restartTbl.STOP_TIME)
                        )
                    )
            SaveMode.UPDATE_EXISTING_STATE ->
                insertHelper.onDuplicateKeyUpdate()
                    .set(restartTbl.STRATEGY_DATA, MySQLDSL.values(restartTbl.STRATEGY_DATA))
                    .set(
                        restartTbl.STOP_TIME,
                        DSL.greatest(
                            DSL.nvl(restartTbl.STOP_TIME, MySQLDSL.values(restartTbl.STOP_TIME)),
                            MySQLDSL.values(restartTbl.STOP_TIME)
                        )
                    )
            SaveMode.IGNORE_EXISTING -> insertHelper.onDuplicateKeyIgnore()
            SaveMode.FAIL_EXISTING -> {}
        }
        insertHelper.execute()
    }
}
