package ru.yandex.direct.autobudget.restart.repository

import com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule
import com.fasterxml.jackson.module.kotlin.readValue
import com.fasterxml.jackson.module.kotlin.registerKotlinModule
import org.jooq.DSLContext
import org.jooq.Record
import org.jooq.SelectFinalStep
import org.jooq.SelectForUpdateStep
import org.jooq.impl.DSL
import org.jooq.util.mysql.MySQLDSL
import org.springframework.stereotype.Component
import ru.yandex.direct.autobudget.restart.model.PackageStrategyDto
import ru.yandex.direct.autobudget.restart.service.StrategyState
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapperhelper.InsertHelper
import ru.yandex.direct.utils.JsonUtils

data class StrategyRestartData(
    val strategyId: Long,
    val strategyData: PackageStrategyDto,
    val times: RestartTimes,
    val state: StrategyState
)

@Component
open class PackageStrategyAutobudgetRestartRepository(
    private val dslProvider: DslContextProvider
) {
    private val restartTbl = Tables.STRATEGY_AUTOBUDGET_RESTART
    private val mapper = ObjectMapper()
        .registerKotlinModule()
        .registerModule(JavaTimeModule())
        .registerModule(JsonUtils.createBigDecimalModule())
        .disable(FAIL_ON_UNKNOWN_PROPERTIES)

    fun getAutobudgetRestartData(shard: Int, strategyIds: List<Long>): List<StrategyRestartData> {
        return getAutobudgetRestartData(dslProvider.ppc(shard), strategyIds, false)
    }

    fun getAutobudgetRestartData(
        dslContext: DSLContext,
        strategyIds: List<Long>,
        withUpdateLock: Boolean
    ): List<StrategyRestartData> {
        var select = dslContext.select(
            restartTbl.STRATEGY_ID,
            restartTbl.RESTART_TIME, restartTbl.SOFT_RESTART_TIME, restartTbl.RESTART_REASON,
            restartTbl.STRATEGY_DATA,
            restartTbl.STOP_TIME
        ).from(restartTbl)
            .where(restartTbl.STRATEGY_ID.`in`(strategyIds)) as SelectFinalStep<Record>
        if (withUpdateLock) {
            select = (select as SelectForUpdateStep).forUpdate()
        }

        return select
            .map { it.into(restartTbl.recordType) }
            .map {
                StrategyRestartData(
                    strategyId = it.strategyId,
                    times = RestartTimes(
                        restartTime = it.restartTime,
                        softRestartTime = it.softRestartTime,
                        restartReason = it.restartReason,
                    ),
                    strategyData = mapper.readValue(it.strategyData),
                    state = StrategyState(it.stopTime)
                )
            }
    }

    fun saveAutobudgetRestartData(
        dslContext: DSLContext,
        data: List<StrategyRestartData>,
        doUpdateOnDuplicate: Boolean = true
    ) {
        if (data.isEmpty())
            return
        val insertHelper = InsertHelper(dslContext, restartTbl)
        data.forEach {
            insertHelper.set(restartTbl.STRATEGY_ID, it.strategyId)
                .set(restartTbl.RESTART_TIME, it.times.restartTime)
                .set(restartTbl.SOFT_RESTART_TIME, it.times.softRestartTime)
                .set(restartTbl.RESTART_REASON, it.times.restartReason)
                .set(restartTbl.STRATEGY_DATA, mapper.writeValueAsString(it.strategyData))
                .set(restartTbl.STOP_TIME, it.state.stopTime)
                .newRecord()
        }
        if (doUpdateOnDuplicate) {
            insertHelper.onDuplicateKeyUpdate()
                .set(restartTbl.RESTART_TIME, MySQLDSL.values(restartTbl.RESTART_TIME))
                .set(restartTbl.SOFT_RESTART_TIME, MySQLDSL.values(restartTbl.SOFT_RESTART_TIME))
                .set(restartTbl.RESTART_REASON, MySQLDSL.values(restartTbl.RESTART_REASON))
                .set(restartTbl.STRATEGY_DATA, MySQLDSL.values(restartTbl.STRATEGY_DATA))
                .set(
                    restartTbl.STOP_TIME,
                    DSL.greatest(
                        DSL.nvl(restartTbl.STOP_TIME, MySQLDSL.values(restartTbl.STOP_TIME)),
                        MySQLDSL.values(restartTbl.STOP_TIME)
                    )
                )
        } else {
            insertHelper.onDuplicateKeyIgnore()
        }
        insertHelper.execute()
    }
}
