package ru.yandex.direct.autobudget.restart.service

import org.springframework.stereotype.Component
import ru.yandex.direct.autobudget.restart.repository.RestartTimes
import ru.yandex.direct.autobudget.restart.service.RestartDecision.Companion.fullRestart
import ru.yandex.direct.autobudget.restart.service.RestartDecision.Companion.noRestart
import ru.yandex.direct.autobudget.restart.service.RestartDecision.Companion.softRestart
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyName
import ru.yandex.direct.common.db.PpcPropertyNames.AUTOBUDGET_RESTART_AUTOBUDGET_AVG_BID_MARGIN
import ru.yandex.direct.common.db.PpcPropertyNames.AUTOBUDGET_RESTART_AUTOBUDGET_AVG_CPA_MARGIN
import ru.yandex.direct.common.db.PpcPropertyNames.AUTOBUDGET_RESTART_AUTOBUDGET_AVG_CPM_MARGIN
import ru.yandex.direct.common.db.PpcPropertyNames.AUTOBUDGET_RESTART_PAUSE_SECONDS_MARGIN
import ru.yandex.direct.common.db.PpcPropertyNames.ENABLE_NEW_CPA_AUTOBUDGET_RESTART_LOGIC_ORDERIDS
import ru.yandex.direct.common.db.PpcPropertyNames.ENABLE_NEW_CPA_AUTOBUDGET_RESTART_LOGIC_PERCENT
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_WEEK_BUNDLE
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.utils.DateTimeUtils
import java.math.BigDecimal
import java.math.BigDecimal.ZERO
import java.math.RoundingMode
import java.time.Duration
import java.time.Instant
import java.time.LocalDate
import java.time.LocalDateTime

/**
 * Тип рестарта стратегии
 */
enum class RestartType {
    NO_RESTART, FULL, SOFT
}

/**
 * ПРичина рестарта стратегии
 * При добавлении сюда нового значения, нужно добавить его в и в https://a.yandex-team.ru/arc/trunk/arcadia/grut/libs/proto/objects/campaign_v2.proto?rev=r9201955#L1
 */
enum class Reason {
    INIT,
    EMPTY,
    BS_RESTART,
    AUTOBUDGET_START,
    DAY_BUDGET_START,
    DAY_BUDGET_CHANGED,
    CHANGED_PAID_ACTION,
    SUM_CHANGED_FOR_PAID_ACTIONS,
    CHANGED_CPA_FOR_PAID_ACTIONS,
    OPTIMIZE_TYPE_CHANGED,
    CHANGED_STRATEGY_NAME,
    CHANGED_START_TIME_TO_PAST,
    CHANGED_START_TIME_TO_FUTURE,
    START_TIME_IN_FUTURE,
    CHANGED_TIME_TARGET,
    CHANGED_AUTOBUDGET_SUM,
    CHANGED_AUTOBUDGET_BID,
    CHANGED_LIMIT_CLICKS,
    CHANGED_AVG_CPA,
    CHANGED_AVG_CPM,
    CHANGED_AVG_CPM_PERIOD,
    CHANGED_AVG_CPV,
    CHANGED_AVG_CPV_PERIOD,
    RESTART_AFTER_MONEY_PAUSE,
    RESTART_AFTER_STOP_PAUSE,
    CPC_HOLD_ENABLED,
    CHANGED_FINISH_TIME_TO_FUTURE,
    OLD_RESTART_FOR_PERIOD_CAMPAIGN,
    NET_CPC_OPTIMIZE_AFTER_DAY_BUDGET,
}

/**
 * Решение о рестарте стратегии
 */
data class RestartDecision(
    val type: RestartType,
    val reason: Reason = Reason.EMPTY,
    val time: LocalDateTime? = null,
) {
    companion object {
        // 2038-01-19T06:14:07
        val MAX_LOCAL_DATE_TIME: LocalDateTime =
            DateTimeUtils.instantToMoscowDateTime(Instant.ofEpochSecond(Int.MAX_VALUE.toLong()))

        fun noRestart() =
            RestartDecision(RestartType.NO_RESTART, Reason.EMPTY)

        fun fullRestart(reason: Reason, time: LocalDateTime? = null): RestartDecision {
            return RestartDecision(RestartType.FULL, reason, getLimitedTime(time))
        }

        fun softRestart(reason: Reason, time: LocalDateTime? = null) =
            RestartDecision(RestartType.SOFT, reason, getLimitedTime(time))

        /**
         * Функция, которая ограничивает время максимальным timestamp
         * Это нужно, т.к. в базе в таблице camp_autobudget_restart колонки restart_time и sort_restart_time
         * имеют тип timestamp, и при попытке записать туда большее значение происходит падение
         */
        private fun getLimitedTime(time: LocalDateTime? = null): LocalDateTime? {
            return if (time?.isAfter(MAX_LOCAL_DATE_TIME) == true)
                MAX_LOCAL_DATE_TIME
            else time
        }
    }
}

// 4%
const val DEFAULT_MARGIN = 0.04

// 4 часа
const val DEFAULT_PAUSE_SECONDS_MARGIN = 4 * 60 * 60

@Component
open class AutobudgetRestartCalculator(
    private val propertiesSupport: PpcPropertiesSupport
) {
    val autoBudgetAvgBidMargin = property(AUTOBUDGET_RESTART_AUTOBUDGET_AVG_BID_MARGIN)
    val autoBudgetAvgCpmMargin = property(AUTOBUDGET_RESTART_AUTOBUDGET_AVG_CPM_MARGIN)
    val autoBudgetAvgCpaMargin = property(AUTOBUDGET_RESTART_AUTOBUDGET_AVG_CPA_MARGIN)
    val enabledNewCpaRestartLogicForOrderIds = property(ENABLE_NEW_CPA_AUTOBUDGET_RESTART_LOGIC_ORDERIDS)
    val enabledNewCpaRestartLogicForPercent = property(ENABLE_NEW_CPA_AUTOBUDGET_RESTART_LOGIC_PERCENT)

    val pauseMargin = property(AUTOBUDGET_RESTART_PAUSE_SECONDS_MARGIN)

    fun compareStrategy(): RestartDecision = RestartDecision(RestartType.FULL)

    fun compareStrategy(
        old: StrategyData?,
        new: StrategyData,
        state: StrategyState,
        times: RestartTimes?,
        orderId: Long?
    ): RestartDecision {
        val today = LocalDate.now()
        return when {
            old == null ->
                when {
                    new.effectiveStartTime > today ->
                        fullRestart(Reason.START_TIME_IN_FUTURE, new.effectiveStartTime.atStartOfDay())

                    else ->
                        fullRestart(Reason.INIT)
                }

            new.isAutoBudget || new.isDayBudget ->
                when {
                    // обработка start_time в начале, потому что время может выставляться в будущем
                    old.effectiveStartTime > today && new.effectiveStartTime <= today ->
                        fullRestart(Reason.CHANGED_START_TIME_TO_PAST)

                    old.effectiveStartTime <= today && new.effectiveStartTime > today ->
                        fullRestart(Reason.CHANGED_START_TIME_TO_FUTURE, new.effectiveStartTime.atStartOfDay())

                    new.effectiveStartTime > today ->
                        fullRestart(Reason.START_TIME_IN_FUTURE, new.effectiveStartTime.atStartOfDay())

                    // кампания была запущена сменой finish_time
                    old.effectiveFinishTime < today && new.effectiveFinishTime >= today && new.startTime <= today
                        && !new.isFixCpmStrategy ->
                        fullRestart(Reason.CHANGED_FINISH_TIME_TO_FUTURE)

                    new.isAutoBudget ->
                        compareAutoBudgetStrategy(old, new, state, times, orderId)

                    new.isDayBudget ->
                        compareDayBudgetStrategy(old, new)

                    else ->
                        throw IllegalStateException("Strategy is not day_budget / auto_budget")
                }

            else ->
                compareManualStrategy(old, new)
        }
    }

    private fun compareManualStrategy(
        old: StrategyData,
        new: StrategyData
    ): RestartDecision =
        when {
            new.platform == CampaignsPlatform.BOTH && new.enableCpcHold
                && !(old.platform == CampaignsPlatform.BOTH && old.enableCpcHold) ->
                fullRestart(Reason.CPC_HOLD_ENABLED)

            old.isDayBudget && new.netCpcOptimize ->
                softRestart(Reason.NET_CPC_OPTIMIZE_AFTER_DAY_BUDGET)

            else ->
                noRestart()
        }

    private fun compareDayBudgetStrategy(
        old: StrategyData,
        new: StrategyData
    ): RestartDecision =
        when {
            !old.isDayBudget ->
                fullRestart(Reason.DAY_BUDGET_START)

            changed(old.dayBudget, new.dayBudget) ->
                softRestart(Reason.DAY_BUDGET_CHANGED)

            else ->
                noRestart()
        }

    private fun compareAutoBudgetStrategy(
        old: StrategyData,
        new: StrategyData,
        state: StrategyState,
        times: RestartTimes?,
        orderId: Long?,
    ): RestartDecision {
        val now = LocalDateTime.now()
        val pauseBorderTime = now.minusSeconds(pauseMargin.getOrDefault(DEFAULT_PAUSE_SECONDS_MARGIN).toLong())
        val isCpmPeriodStrategy = new.isCpmPeriodStrategy
        val isCpmStrategy = new.isCpmStrategy
        val isCpvPeriodStrategy = new.strategyName == StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD
        val isCpvStrategy = new.strategyName == StrategyName.AUTOBUDGET_AVG_CPV ||
            new.strategyName == StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD
        val isEnabledNewCpaLogicById = orderId != null && enabledNewCpaRestartLogicForOrderIds
            .getOrDefault(emptySet()).contains(orderId)
        val newCpaRestartLogicPercent = enabledNewCpaRestartLogicForPercent.getOrDefault(0)
        val isEnabledNewCpaLogicByPercent = newCpaRestartLogicPercent >= 100 ||
            orderId != null && orderId % 100 < newCpaRestartLogicPercent
        val isEnabledNewCpaLogic = isEnabledNewCpaLogicById || isEnabledNewCpaLogicByPercent

        return when {
            !old.isAutoBudget ->
                fullRestart(Reason.AUTOBUDGET_START)

            old.isAutoBudgetPaidActions != new.isAutoBudgetPaidActions ->
                fullRestart(Reason.CHANGED_PAID_ACTION)

            // поступление средств после долгого простоя
            !isCpmPeriodStrategy && !isCpvPeriodStrategy && new.hasMoney && !old.hasMoney
                && (state.stopTime?.isBefore(pauseBorderTime) ?: false) ->
                fullRestart(Reason.RESTART_AFTER_MONEY_PAUSE)

            // включение после долгой остановки
            !isCpmPeriodStrategy && !isCpvPeriodStrategy && new.statusShow && !old.statusShow
                && (state.stopTime?.isBefore(pauseBorderTime) ?: false) ->
                fullRestart(Reason.RESTART_AFTER_STOP_PAUSE)

            new.isAutoBudgetPaidActions ->
                when {
                    changed(old.autoBudgetSum, new.autoBudgetSum) ->
                        fullRestart(Reason.SUM_CHANGED_FOR_PAID_ACTIONS)
                    isEnabledNewCpaLogic && changed(old.avgCpa, new.avgCpa, autoBudgetAvgCpaMargin) ->
                        softRestart(Reason.CHANGED_CPA_FOR_PAID_ACTIONS)
                    else -> noRestart()
                }

            old.autoBudgetOptimizeType != new.autoBudgetOptimizeType ->
                fullRestart(Reason.OPTIMIZE_TYPE_CHANGED)

            // изменение недельного бюджета
            changed(old.autoBudgetSum, new.autoBudgetSum) ->
                fullRestart(Reason.CHANGED_AUTOBUDGET_SUM)

            // изменение cpc на недельной кампании
            changed(old.avgBid, new.avgBid, autoBudgetAvgBidMargin) ->
                fullRestart(Reason.CHANGED_AUTOBUDGET_BID)

            // изменение среднего cpa на недельной стратегии с оплатой за клики и оптимизации конверсий
            isEnabledNewCpaLogic && changed(old.avgCpa, new.avgCpa, autoBudgetAvgCpaMargin) ->
                softRestart(Reason.CHANGED_AVG_CPA)

            // изменение целевого кол-ва кликов на продукте "пакет кликов"
            // продукт закопан тут - https://st.yandex-team.ru/DIRECT-118934
            new.strategyName == AUTOBUDGET_WEEK_BUNDLE && old.limitClicks != new.limitClicks ->
                fullRestart(Reason.CHANGED_LIMIT_CLICKS)

            // (охватная реклама) недельная cpm стратегия и значительное изменение avgCpm
            isCpmStrategy && !isCpmPeriodStrategy && changed(old.avgCpm, new.avgCpm, autoBudgetAvgCpmMargin) ->
                fullRestart(Reason.CHANGED_AVG_CPM)

            // (охватная реклама) недельная cpv стратегия и значительное изменение avgCpv
            isCpvStrategy && !isCpvPeriodStrategy && changed(old.avgCpv, new.avgCpv, autoBudgetAvgCpmMargin) ->
                fullRestart(Reason.CHANGED_AVG_CPV)

            // (охватная реклама) изменение времени начала кампании на периодной стратегии
            new.isCustomPeriodStrategy && times != null
                && times.restartTime < new.effectiveStartTime.atStartOfDay() ->
                fullRestart(Reason.OLD_RESTART_FOR_PERIOD_CAMPAIGN)

            // (охватная реклама) периодная cpm стратегия и строгое изменение avgCpm
            isCpmStrategy && isCpmPeriodStrategy && changed(old.avgCpm, new.avgCpm) ->
                softRestart(Reason.CHANGED_AVG_CPM_PERIOD)

            // (охватная реклама) периодная cpm стратегия и строгое изменение avgCpv
            isCpvStrategy && isCpvPeriodStrategy && changed(old.avgCpv, new.avgCpv) ->
                softRestart(Reason.CHANGED_AVG_CPV_PERIOD)

            else -> noRestart()
        }
    }

    private fun changed(old: BigDecimal, new: BigDecimal, marginProp: PpcProperty<Double>) =
        if (old == ZERO) {
            new > ZERO
        } else {
            val margin = marginProp.getOrDefault(DEFAULT_MARGIN)
            // результат деления наследует точность от первого аргумента
            ((new - old).abs().setScale(6, RoundingMode.HALF_UP) / old).toDouble() > margin
        }

    private fun changed(old: BigDecimal, new: BigDecimal) =
        old.minus(new).abs() > Currencies.EPSILON

    private fun <T> property(name: PpcPropertyName<T>) =
        propertiesSupport.get(name, Duration.ofMinutes(1))
}
