package ru.yandex.direct.autobudget.restart.service

import org.jooq.DSLContext
import org.springframework.stereotype.Component
import ru.yandex.direct.autobudget.restart.model.CampStrategyRestartData
import ru.yandex.direct.autobudget.restart.model.CampStrategyRestartResultSuccess
import ru.yandex.direct.autobudget.restart.model.StrategyDto
import ru.yandex.direct.autobudget.restart.repository.CampRestartData
import ru.yandex.direct.autobudget.restart.repository.CampaignAutobudgetRestartRepository
import ru.yandex.direct.autobudget.restart.repository.RestartTimes
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.temporal.ChronoUnit

data class CampaignAutobudgetRestartContainer(
    val restartResult: CampStrategyRestartResultSuccess,
    val restartDbData: CampRestartData,
)

@Component
open class AutobudgetRestartService(
    private val calculator: AutobudgetRestartCalculator,
    private val repository: CampaignAutobudgetRestartRepository,
    private val dslContextProvider: DslContextProvider,
) {
    companion object {
        val OLD_START_TIME = LocalDate.parse("2000-01-01")
    }

    fun calculateRestartsAndSave(
        shard: Int,
        chunk: List<CampStrategyRestartData>
    ): List<CampaignAutobudgetRestartContainer> {
        val result = mutableListOf<CampaignAutobudgetRestartContainer>()
        dslContextProvider.ppcTransaction(shard) { configuration ->
            result += calculateRestartsAndSave(configuration.dsl(), chunk)
        }
        return result
    }

    fun calculateRestartsAndSave(
        dslContext: DSLContext,
        chunk: List<CampStrategyRestartData>
    ): List<CampaignAutobudgetRestartContainer> {
        val result = mutableListOf<CampaignAutobudgetRestartContainer>()
        val dbInfoMap = repository.getAutobudgetRestartData(dslContext, chunk.map { it.cid }, true)
            .associateBy { it.cid }

        val restartContainer = calculateRestarts(chunk, dbInfoMap.values)
        val dbWrites = mutableListOf<CampRestartData>()
        restartContainer.forEach {
            if (it.restartDbData != dbInfoMap[it.restartDbData.cid]) {
                dbWrites.add(it.restartDbData)
            }
            result.add(it)
        }

        repository.saveAutobudgetRestartData(dslContext, dbWrites)
        return result
    }

    fun calculateRestarts(
        chunk: List<CampStrategyRestartData>,
        dbInfo: Collection<CampRestartData>
    ): List<CampaignAutobudgetRestartContainer> {
        val dbInfoMap = dbInfo.associateBy { it.cid }
        val result = mutableListOf<CampaignAutobudgetRestartContainer>()
        for (camp in chunk) {
            val dbData = dbInfoMap[camp.cid]

            val state = dbData?.state ?: StrategyState()
            val oldStrategy = dbData?.let { StrategyData(it.strategyData) }
            val newStrategy = StrategyData(camp.strategyDto)

            val decision = calculator.compareStrategy(
                oldStrategy, newStrategy, state, dbData?.times, dbData?.orderId)

            val ret = applyCalcDecision(camp.cid, dbData, decision)
            val newDbData = CampRestartData(
                cid = camp.cid,
                orderId = dbData?.orderId,
                times = RestartTimes(
                    restartTime = ret.restartTime,
                    softRestartTime = ret.softRestartTime,
                    restartReason = ret.restartReason
                ),
                strategyData = camp.strategyDto,
                state = calcState(state, oldStrategy, newStrategy),
            )

            result.add(CampaignAutobudgetRestartContainer(ret, newDbData))
        }
        return result
    }

    fun getActualRestartTimes(
        shard: Int,
        strategyByCid: Map<Long, StrategyDto>
    ): List<CampStrategyRestartResultSuccess> {
        val dbInfoMap = repository.getAutobudgetRestartData(shard, strategyByCid.keys.toList())
            .associateBy { it.cid }
        val result = mutableListOf<CampStrategyRestartResultSuccess>()
        for (entry in strategyByCid) {
            val newStrategy = StrategyData(entry.value)
            val dbData = dbInfoMap[entry.key]
            val state = dbData?.state ?: StrategyState()
            val oldStrategy = dbData?.let { StrategyData(it.strategyData) }
            val decision = calculator.compareStrategy(
                oldStrategy, newStrategy, state, dbData?.times, dbData?.orderId)
            result.add(applyCalcDecision(entry.key, dbData, decision))
        }
        return result
    }

    private fun calcState(state: StrategyState, old: StrategyData?, new: StrategyData): StrategyState =
        if (!new.hasMoney && old?.hasMoney != new.hasMoney)
            StrategyState(stopTime = LocalDateTime.now())
        else if (!new.statusShow && old?.statusShow != new.statusShow)
            StrategyState(stopTime = LocalDateTime.now())
        else
            state

    private fun applyCalcDecision(
        cid: Long,
        dbData: CampRestartData?,
        decision: RestartDecision
    ): CampStrategyRestartResultSuccess {
        val now = LocalDateTime.now().truncatedTo(ChronoUnit.SECONDS)
        val ret = CampStrategyRestartResultSuccess(
            cid = cid,
            restartTime = dbData?.times?.restartTime ?: now,
            softRestartTime = dbData?.times?.softRestartTime ?: now,
            restartReason = dbData?.times?.restartReason ?: Reason.EMPTY.name
        )
        when (decision.type) {
            RestartType.NO_RESTART -> {
            }
            RestartType.SOFT -> {
                ret.softRestartTime = decision.time ?: now
                ret.restartReason = decision.reason.name
            }
            RestartType.FULL -> {
                ret.restartTime = decision.time ?: now
                ret.softRestartTime = decision.time ?: now
                ret.restartReason = decision.reason.name
            }
        }
        return ret
    }
}
