package ru.yandex.direct.autobudget.restart.service

import org.jooq.DSLContext
import org.springframework.stereotype.Service
import ru.yandex.direct.autobudget.restart.model.PackageStrategyDto
import ru.yandex.direct.autobudget.restart.model.PackageStrategyRestartResult
import ru.yandex.direct.autobudget.restart.repository.PackageStrategyAutobudgetRestartRepository
import ru.yandex.direct.autobudget.restart.repository.RestartTimes
import ru.yandex.direct.autobudget.restart.repository.StrategyRestartData
import java.time.LocalDateTime
import java.time.temporal.ChronoUnit
import java.util.function.Supplier

@Service
class AutobudgetStrategyRestartService(
    private val calculator: PackageStrategyAutobudgetRestartCalculator,
    private val repository: PackageStrategyAutobudgetRestartRepository
) {

    fun recalculateAndSaveRestarts(
        dslContext: DSLContext,
        dataList: List<PackageStrategyRestartData>,
        initialRestartsProvider: InitialRestartsProvider,
        withLock: Boolean
    ): List<PackageStrategyRestartResult> {
        val result = mutableListOf<PackageStrategyRestartResult>()
        val restarts = calculateRestarts(dslContext, dataList, initialRestartsProvider, withLock)
        repository.saveAutobudgetRestartData(dslContext, restarts)
        restarts
            .map {
                result.add(
                    PackageStrategyRestartResult(
                        it.strategyId,
                        it.times.restartTime,
                        it.times.softRestartTime,
                        it.times.restartReason,
                        it.state
                    )
                )
            }
        return result
    }

    private fun calculateRestarts(
        dslContext: DSLContext,
        data: List<PackageStrategyRestartData>,
        initialRestartsProvider: InitialRestartsProvider,
        withLock: Boolean
    ): List<StrategyRestartData> {
        val now = LocalDateTime.now().truncatedTo(ChronoUnit.SECONDS)
        val oldRestarts =
            repository.getAutobudgetRestartData(dslContext, data.map(PackageStrategyRestartData::strategyId), withLock)
                .associateBy { it.strategyId }
        return data.map { restartData ->
            val oldRestart = oldRestarts[restartData.strategyId]
            val decision = calculator.compareStrategy(
                old = oldRestart?.strategyData?.let { PackageStrategyData(it) },
                new = PackageStrategyData(restartData.dto),
                state = oldRestart?.state ?: StrategyState(),
                times = oldRestart?.times
            )
            val oldState = oldRestart?.state ?: StrategyState()
            val oldRestartTimes = oldRestart?.times ?: initialRestartsProvider.getRestartTime(restartData.strategyId)
            StrategyRestartData(
                restartData.strategyId,
                restartData.dto,
                restartTimes(now, oldRestartTimes, decision),
                calcState(oldState, oldRestart?.strategyData, restartData.dto, now)
            )
        }
    }

    // oldRestartTimes == null кейс, когда завели публичную стратегию без кампаний.
    // Считаем это рестартом с restartTime = now.
    private fun restartTimes(
        now: LocalDateTime,
        oldRestartTimes: RestartTimes?,
        decision: RestartDecision
    ): RestartTimes =
        when (decision.type) {
            RestartType.FULL ->
                RestartTimes(
                    restartTime = decision.time ?: now,
                    softRestartTime = decision.time ?: now,
                    restartReason = decision.reason.name
                )
            RestartType.SOFT ->
                RestartTimes(
                    restartTime = oldRestartTimes?.restartTime ?: now,
                    softRestartTime = decision.time ?: now,
                    restartReason = decision.reason.name
                )
            RestartType.NO_RESTART ->
                oldRestartTimes ?: RestartTimes(
                    restartTime = now,
                    softRestartTime = now,
                    restartReason = Reason.INIT.name
                )
        }

    private fun calcState(
        state: StrategyState,
        old: PackageStrategyDto?,
        new: PackageStrategyDto,
        now: LocalDateTime
    ): StrategyState =
        if (isSpentAllMoney(old, new))
            StrategyState(stopTime = now)
        else
            state

    private fun isSpentAllMoney(
        old: PackageStrategyDto?,
        new: PackageStrategyDto
    ): Boolean =
        !new.hasMoney && old?.hasMoney == true
}

data class PackageStrategyRestartData(val strategyId: Long, val dto: PackageStrategyDto)

class InitialRestartsProvider(supplier: Supplier<Map<Long, RestartTimes?>>) {
    private val restartTimeByStrategyId by lazy { supplier.get() }

    fun getRestartTime(strategyId: Long): RestartTimes? =
        restartTimeByStrategyId[strategyId]
}

