package ru.yandex.direct.autobudget.restart.service

import org.springframework.stereotype.Component
import ru.yandex.direct.autobudget.restart.repository.RestartTimes
import ru.yandex.direct.autobudget.restart.service.RestartDecision.Companion.fullRestart
import ru.yandex.direct.autobudget.restart.service.RestartDecision.Companion.noRestart
import ru.yandex.direct.autobudget.restart.service.RestartDecision.Companion.softRestart
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyName
import ru.yandex.direct.common.db.PpcPropertyNames.AUTOBUDGET_RESTART_AUTOBUDGET_AVG_BID_MARGIN
import ru.yandex.direct.common.db.PpcPropertyNames.AUTOBUDGET_RESTART_AUTOBUDGET_AVG_CPA_MARGIN
import ru.yandex.direct.common.db.PpcPropertyNames.AUTOBUDGET_RESTART_AUTOBUDGET_AVG_CPM_MARGIN
import ru.yandex.direct.common.db.PpcPropertyNames.AUTOBUDGET_RESTART_PAUSE_SECONDS_MARGIN
import ru.yandex.direct.common.db.PpcPropertyNames.ENABLE_NEW_CPA_AUTOBUDGET_RESTART_LOGIC_PERCENT
import ru.yandex.direct.core.entity.strategy.model.StrategyName
import ru.yandex.direct.core.entity.strategy.model.StrategyName.AUTOBUDGET_WEEK_BUNDLE
import ru.yandex.direct.currency.Currencies
import java.math.BigDecimal
import java.math.BigDecimal.ZERO
import java.math.RoundingMode
import java.time.Duration
import java.time.LocalDate
import java.time.LocalDateTime

@Component
open class PackageStrategyAutobudgetRestartCalculator(
    private val propertiesSupport: PpcPropertiesSupport
) {
    val autoBudgetAvgBidMargin = property(AUTOBUDGET_RESTART_AUTOBUDGET_AVG_BID_MARGIN)
    val autoBudgetAvgCpmMargin = property(AUTOBUDGET_RESTART_AUTOBUDGET_AVG_CPM_MARGIN)
    val autoBudgetAvgCpaMargin = property(AUTOBUDGET_RESTART_AUTOBUDGET_AVG_CPA_MARGIN)
    val pauseMargin = property(AUTOBUDGET_RESTART_PAUSE_SECONDS_MARGIN)
    val enabledNewCpaRestartLogicForPercent = property(ENABLE_NEW_CPA_AUTOBUDGET_RESTART_LOGIC_PERCENT)

    fun compareStrategy(): RestartDecision = RestartDecision(RestartType.FULL)

    fun compareStrategy(
        old: PackageStrategyData?,
        new: PackageStrategyData,
        state: StrategyState,
        times: RestartTimes?,
    ): RestartDecision {
        val today = LocalDate.now()
        return when {
            old == null ->
                // Пакетная стратегия стала публичной
                noRestart()

            new.isAutoBudget || new.isDayBudget ->
                when {
                    //Обработка стратегий с кастомным периодом
                    // обработка start_time в начале, потому что время может выставляться в будущем
                    old.customPeriodStrategyStart.isAfter(today) && !new.customPeriodStrategyStart.isAfter(today) ->
                        fullRestart(Reason.CHANGED_START_TIME_TO_PAST)
                    // Время старта в новой версии определено поэтому `!!`
                    !old.customPeriodStrategyStart.isAfter(today) && new.customPeriodStrategyStart.isAfter(today) ->
                        fullRestart(Reason.CHANGED_START_TIME_TO_FUTURE, new.customPeriodStrategyStart!!.atStartOfDay())
                    //Время старта в новой версии определено поэтому `!!`
                    new.customPeriodStrategyStart.isAfter(today) ->
                        fullRestart(Reason.START_TIME_IN_FUTURE, new.customPeriodStrategyStart!!.atStartOfDay())
                    // стратегия была запущена сменой finish_time
                    old.customPeriodStrategyFinish < today && new.customPeriodStrategyFinish >= today && !new.customPeriodStrategyStart.isAfter(
                        today
                    )
                        && !new.isFixCpmStrategy ->
                        fullRestart(Reason.CHANGED_FINISH_TIME_TO_FUTURE)

                    new.isAutoBudget ->
                        compareAutoBudgetStrategy(old, new, state, times)

                    new.isDayBudget ->
                        compareDayBudgetStrategy(old, new)

                    else ->
                        throw IllegalStateException("Strategy is not day_budget / auto_budget")
                }

            else ->
                noRestart()
        }
    }

    private fun compareDayBudgetStrategy(
        old: PackageStrategyData,
        new: PackageStrategyData
    ): RestartDecision =
        when {
            !old.isDayBudget ->
                fullRestart(Reason.DAY_BUDGET_START)

            changed(old.dayBudget, new.dayBudget) ->
                softRestart(Reason.DAY_BUDGET_CHANGED)

            else ->
                noRestart()
        }

    private fun compareAutoBudgetStrategy(
        old: PackageStrategyData,
        new: PackageStrategyData,
        state: StrategyState,
        times: RestartTimes?
    ): RestartDecision {
        val now = LocalDateTime.now()
        val pauseBorderTime = now.minusSeconds(pauseMargin.getOrDefault(DEFAULT_PAUSE_SECONDS_MARGIN).toLong())
        val isCpmPeriodStrategy = new.isCpmPeriodStrategy
        val isCpmStrategy = new.isCpmStrategy
        val isCpvPeriodStrategy = new.strategyName == StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD
        val isCpvStrategy = new.strategyName == StrategyName.AUTOBUDGET_AVG_CPV ||
            new.strategyName == StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD
        val isEnabledNewCpaLogic = enabledNewCpaRestartLogicForPercent.getOrDefault(0) >= 100

        return when {
            !old.isAutoBudget ->
                fullRestart(Reason.AUTOBUDGET_START)

            old.isAutoBudgetPaidActions != new.isAutoBudgetPaidActions ->
                fullRestart(Reason.CHANGED_PAID_ACTION)

            // поступление средств после долгой остановки
            !isCpmPeriodStrategy && !isCpvPeriodStrategy && new.hasMoney && !old.hasMoney
                && (state.stopTime?.isBefore(pauseBorderTime) ?: false) ->
                fullRestart(Reason.RESTART_AFTER_MONEY_PAUSE)

            new.isAutoBudgetPaidActions ->
                when {
                    changed(old.autoBudgetSum, new.autoBudgetSum) ->
                        fullRestart(Reason.SUM_CHANGED_FOR_PAID_ACTIONS)
                    isEnabledNewCpaLogic && changed(old.avgCpa, new.avgCpa, autoBudgetAvgCpaMargin) ->
                        softRestart(Reason.CHANGED_CPA_FOR_PAID_ACTIONS)
                    else -> noRestart()
                }

            old.autoBudgetOptimizeType != new.autoBudgetOptimizeType ->
                fullRestart(Reason.OPTIMIZE_TYPE_CHANGED)

            // изменение недельного бюджета
            changed(old.autoBudgetSum, new.autoBudgetSum) ->
                fullRestart(Reason.CHANGED_AUTOBUDGET_SUM)

            // изменение cpc на недельной кампании
            changed(old.avgBid, new.avgBid, autoBudgetAvgBidMargin) ->
                fullRestart(Reason.CHANGED_AUTOBUDGET_BID)

            // изменение среднего cpa на недельной стратегии с оплатой за клики и оптимизации конверсий
            isEnabledNewCpaLogic && changed(old.avgCpa, new.avgCpa, autoBudgetAvgCpaMargin) ->
                softRestart(Reason.CHANGED_AVG_CPA)

            // изменение целевого кол-ва кликов на продукте "пакет кликов"
            // продукт закопан тут - https://st.yandex-team.ru/DIRECT-118934
            new.strategyName == AUTOBUDGET_WEEK_BUNDLE && old.limitClicks != new.limitClicks ->
                fullRestart(Reason.CHANGED_LIMIT_CLICKS)

            // (охватная реклама) недельная cpm стратегия и значительное изменение avgCpm
            isCpmStrategy && !isCpmPeriodStrategy && changed(old.avgCpm, new.avgCpm, autoBudgetAvgCpmMargin) ->
                fullRestart(Reason.CHANGED_AVG_CPM)

            // (охватная реклама) недельная cpv стратегия и значительное изменение avgCpv
            isCpvStrategy && !isCpvPeriodStrategy && changed(old.avgCpv, new.avgCpv, autoBudgetAvgCpmMargin) ->
                fullRestart(Reason.CHANGED_AVG_CPV)

            // (охватная реклама) изменение времени начала кампании на периодной стратегии
            new.isCustomPeriodStrategy && times != null && new.customPeriodStrategyStart != null
                && times.restartTime < new.customPeriodStrategyStart!!.atStartOfDay() ->
                fullRestart(Reason.OLD_RESTART_FOR_PERIOD_CAMPAIGN)

            // (охватная реклама) периодная cpm стратегия и строгое изменение avgCpm
            isCpmStrategy && isCpmPeriodStrategy && changed(old.avgCpm, new.avgCpm) ->
                softRestart(Reason.CHANGED_AVG_CPM_PERIOD)

            // (охватная реклама) периодная cpm стратегия и строгое изменение avgCpv
            isCpvStrategy && isCpvPeriodStrategy && changed(old.avgCpv, new.avgCpv) ->
                softRestart(Reason.CHANGED_AVG_CPV_PERIOD)

            else -> noRestart()
        }
    }

    private fun changed(old: BigDecimal, new: BigDecimal, marginProp: PpcProperty<Double>) =
        if (old == ZERO) {
            new > ZERO
        } else {
            val margin = marginProp.getOrDefault(DEFAULT_MARGIN)
            // результат деления наследует точность от первого аргумента
            ((new - old).abs().setScale(6, RoundingMode.HALF_UP) / old).toDouble() > margin
        }

    private fun changed(old: BigDecimal, new: BigDecimal) =
        old.minus(new).abs() > Currencies.EPSILON

    private fun <T> property(name: PpcPropertyName<T>) =
        propertiesSupport.get(name, Duration.ofMinutes(1))

    private fun LocalDate?.isAfter(date: LocalDate): Boolean =
        this?.isAfter(date) == true
}
