package ru.yandex.direct.autobudget.restart.service

import ru.yandex.direct.autobudget.restart.model.PackageStrategyDto
import ru.yandex.direct.core.entity.strategy.model.StrategyName
import java.math.BigDecimal
import java.time.LocalDate

class PackageStrategyData(private val dto: PackageStrategyDto) {
    val strategyName: StrategyName

    init {
        this.strategyName = dto.strategyType
    }

    val dayBudget get() = dto.dayBudget ?: BigDecimal.ZERO
    val autoBudgetSum get() = dto.sum ?: BigDecimal.ZERO
    val enableCpcHold get() = dto.enableCpcHold ?: false
    val payForConversion get() = dto.isPayForConversionEnabled ?: false
    val limitClicks get() = dto.limitClicks ?: 0
    val avgCpm get() = dto.avgCpm ?: BigDecimal.ZERO
    val avgCpv get() = dto.avgCpv ?: BigDecimal.ZERO
    val avgBid get() = dto.avgBid ?: BigDecimal.ZERO
    val avgCpa get() = dto.avgCpa ?: BigDecimal.ZERO
    val roiCoef get() = dto.roiCoef ?: 0
    val hasMoney get() = dto.hasMoney
    val isPublic get() = dto.isPublic

    val isAutoBudget
        get() = strategyName != StrategyName.DEFAULT_ && strategyName != StrategyName.CPM_DEFAULT

    val isDayBudget
        get() = !isAutoBudget && dayBudget > BigDecimal.ZERO

    val customPeriodStrategyStart
        get() =
            if (isCustomPeriodStrategy) {
                dto.start
            } else {
                null
            }

    val customPeriodStrategyFinish
        get() =
            if (isCustomPeriodStrategy) {
                dto.finish
            } else {
                null
            } ?: LocalDate.MAX

    val isCpaStrategy
        get() =
            when (strategyName) {
                StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER, StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP,
                StrategyName.AUTOBUDGET_AVG_CPA, StrategyName.AUTOBUDGET_AVG_CPI -> true
                else -> false
            }

    val isCpmStrategy
        get() =
            when (strategyName) {
                StrategyName.AUTOBUDGET_MAX_IMPRESSIONS,
                StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD,
                StrategyName.AUTOBUDGET_MAX_REACH,
                StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD,
                StrategyName.CPM_DEFAULT -> true
                else -> false
            }

    val isCpmPeriodStrategy
        get() = when (strategyName) {
            StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD,
            StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD,
            StrategyName.PERIOD_FIX_BID -> true
            else -> false
        }

    val isCustomPeriodStrategy
        get() = when (strategyName) {
            StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD,
            StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD,
            StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD,
            StrategyName.PERIOD_FIX_BID -> true
            else -> false
        }

    val isFixCpmStrategy
        get() = strategyName == StrategyName.PERIOD_FIX_BID

    val isRoiStrategy
        get() = strategyName == StrategyName.AUTOBUDGET_ROI

    val isCpiStrategy
        get() = strategyName == StrategyName.AUTOBUDGET_AVG_CPI

    val isAutoBudgetPaidActions
        get() = isCpaStrategy && payForConversion

    val autoBudgetOptimizeRF
        get() = when (strategyName) {
            StrategyName.AUTOBUDGET_MAX_REACH, StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD, StrategyName.AUTOBUDGET_MAX_IMPRESSIONS, StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD -> true
            else -> false
        }

    val hasGoals
        get() = autoBudgetGoalId != null

    val autoBudgetOptimizeType
        get() = when {
            autoBudgetOptimizeRF -> 3
            hasGoals && autobudgetRoiLevel > 0 -> 2
            hasGoals -> 1
            else -> 0
        }

    val autoBudgetGoalId: Long?
        get() = when {
            isAutoBudget && dto.goalId != null -> dto.goalId
            isCpiStrategy -> BS_AVG_CPI_GOAL_ID
            else -> null
        }

    val autobudgetRoiLevel: Double
        get() = when {
            isRoiStrategy && dto.roiCoef != null -> 1.0 / (1 + dto.roiCoef.toInt())
            else -> 0.0
        }
}

