package ru.yandex.direct.autobudget.restart.service

import ru.yandex.direct.autobudget.restart.StrategyUtils
import ru.yandex.direct.autobudget.restart.model.StrategyDto
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import java.math.BigDecimal
import java.math.BigDecimal.ZERO
import java.time.LocalDate
import java.time.LocalDateTime

const val BS_AVG_CPI_GOAL_ID = 4L;

/**
 * Обёртка для StrategyDto - дефолты и классификация стратегий по типам
 */
class StrategyData {
    private val dto: StrategyDto

    val strategyName: StrategyName
    val manualStrategy: CampOptionsStrategy?
    val platform: CampaignsPlatform
    val dayBudget get() = dto.dayBudget ?: ZERO
    val startTime get() = dto.startTime
    val statusShow get() = dto.statusShow
    val autoBudgetSum get() = dto.autoBudgetSum ?: ZERO
    val timeTarget get() = dto.timeTarget
    val enableCpcHold get() = dto.enableCpcHold ?: false
    val payForConversion get() = dto.payForConversion ?: false
    val limitClicks get() = dto.limitClicks ?: 0
    val avgCpm get() = dto.avgCpm ?: ZERO
    val avgCpv get() = dto.avgCpv ?: ZERO
    val avgBid get() = dto.avgBid ?: ZERO
    val avgCpa get() = dto.avgCpa ?: ZERO
    val roiCoef get() = dto.roiCoef ?: ZERO
    val hasMoney get() = dto.hasMoney ?: false

    constructor(dto: StrategyDto) {
        this.dto = dto
        this.strategyName =
            if (dto.strategy.isEmpty()) StrategyName.DEFAULT_
            else StrategyUtils.parseStrategyName(dto.strategy)
        this.platform =
            if (dto.platform.isNullOrEmpty()) CampaignsPlatform.BOTH
            else StrategyUtils.parsePlatform(dto.platform)
        this.manualStrategy =
            if (dto.manualStrategy.isNullOrEmpty()) null
            else StrategyUtils.parseManualStrategy(dto.manualStrategy)

    }

    val isAutoBudget
        get() = strategyName != StrategyName.DEFAULT_ && strategyName != StrategyName.CPM_DEFAULT

    val isDayBudget
        get() = !isAutoBudget && dayBudget > ZERO

    val netCpcOptimize
        get() = !isAutoBudget && manualStrategy != CampOptionsStrategy.DIFFERENT_PLACES
                && (dto.enableCpcHold ?: false) && platform != CampaignsPlatform.SEARCH

    val effectiveStartTime
        get() =
            when (strategyName) {
                StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD,
                StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD,
                StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD,
                StrategyName.PERIOD_FIX_BID
                -> dto.strategyStart

                else
                -> dto.startTime
            } ?: dto.startTime

    val effectiveFinishTime
        get() =
            when (strategyName) {
                StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD,
                StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD,
                StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD,
                StrategyName.PERIOD_FIX_BID
                -> dto.strategyFinish

                else
                -> dto.finishTime
            } ?: dto.finishTime ?: LocalDate.MAX

    val isCpaStrategy
        get() =
            when (strategyName) {
                StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER, StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP,
                StrategyName.AUTOBUDGET_AVG_CPA, StrategyName.AUTOBUDGET_AVG_CPI -> true
                else -> false
            }

    val isCpmStrategy
        get() =
            when (strategyName) {
                StrategyName.AUTOBUDGET_MAX_IMPRESSIONS, StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD,
                StrategyName.AUTOBUDGET_MAX_REACH, StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD,
                StrategyName.CPM_DEFAULT -> true
                else -> false
            }

    val isCpmPeriodStrategy
        get() = when (strategyName) {
            StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD, StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD,
            StrategyName.PERIOD_FIX_BID -> true
            else -> false
        }

    val isCustomPeriodStrategy
        get() = when (strategyName) {
            StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD,
            StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD,
            StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD -> true
            else -> false
        }

    val isFixCpmStrategy
        get() = strategyName == StrategyName.PERIOD_FIX_BID

    val isRoiStrategy
        get() = strategyName == StrategyName.AUTOBUDGET_ROI

    val isCpiStrategy
        get() = strategyName == StrategyName.AUTOBUDGET_AVG_CPI


    val isAutoBudgetPaidActions
        get() = isCpaStrategy && payForConversion

    val autoBudgetOptimizeRF
        get() = when (strategyName) {
            StrategyName.AUTOBUDGET_MAX_REACH, StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD, StrategyName.AUTOBUDGET_MAX_IMPRESSIONS, StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD -> true
            else -> false
        }

    val hasGoals
        get() = autoBudgetGoalId != null || (dto.hasCombinedGoals ?: false)

    val autoBudgetOptimizeType
        get() = when {
            autoBudgetOptimizeRF -> 3
            hasGoals && autobudgetRoiLevel > ZERO -> 2
            hasGoals -> 1
            else -> 0
        }

    val autoBudgetGoalId: Long?
        get() = when {
            isAutoBudget && dto.goalId != null -> dto.goalId
            isCpiStrategy -> BS_AVG_CPI_GOAL_ID
            else -> null
        }

    val autobudgetRoiLevel: BigDecimal
        get() = when {
            isRoiStrategy -> BigDecimal.ONE / (BigDecimal.ONE + dto.roiCoef!!)
            else -> BigDecimal.ZERO
        }
}

// дополнительные данные, которые сохраняем в отдельных полях таблицы с рестартами
data class StrategyState(
    var stopTime: LocalDateTime? = null
)
