package ru.yandex.direct.dbutil.model;

import java.io.IOException;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;

/**
 * Модель для ClientId.
 * Её использование позволяет избежать ошибочного указания uid вместо clientId (оба по смыслу представлены типом long).
 * <p>
 * Реализует {@link Comparable} для удобства сортировки не-{@code null} значений.
 */
@ParametersAreNonnullByDefault
@JsonSerialize(using = ClientId.Serializer.class)
@JsonDeserialize(using = ClientId.Deserializer.class)
public class ClientId implements Comparable<ClientId> {
    private final long id;

    private ClientId(long id) {
        this.id = id;
    }

    public static ClientId fromLong(long id) {
        return new ClientId(id);
    }

    @Nullable
    public static ClientId fromNullableLong(@Nullable Long id) {
        if (id == null) {
            return null;
        }
        return new ClientId(id);
    }

    public long asLong() {
        return id;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ClientId clientId = (ClientId) o;
        return id == clientId.id;
    }

    @Override
    public int hashCode() {
        return Long.hashCode(id);
    }

    @Override
    public String toString() {
        return Long.toString(id);
    }

    @Override
    public int compareTo(ClientId o) {
        return Long.compare(this.id, o.id);
    }

    /**
     * Миниатюрный JSON-сериализатор для {@link ClientId}
     * Сериализует значение как число, {@code null} как {@code null}
     */
    public static class Serializer extends JsonSerializer<ClientId> {
        @Override
        public void serialize(ClientId value, JsonGenerator gen, SerializerProvider provider) throws IOException {
            if (value == null) {
                gen.writeNull();
            } else {
                gen.writeNumber(value.asLong());
            }
        }
    }

    /**
     * Миниатюрный JSON-десериализатор для {@link ClientId}
     * Судя по тестам - десериализация прекрасно работает "из коробки", без использования этого класса
     */
    public static class Deserializer extends JsonDeserializer<ClientId> {
        @Override
        public ClientId deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
            JsonNode node = p.getCodec().readTree(p);
            return node.isNull() ? null : fromLong(node.asLong());
        }
    }
}
