package ru.yandex.direct.dbutil.model;

import java.util.Objects;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.base.Preconditions;

/**
 * Исключающее ИЛИ из логина и uid.
 */
public class LoginOrUid {
    @Nullable
    private final String login;

    @Nullable
    private final Long uid;

    private LoginOrUid(@Nullable String login, @Nullable Long uid) {
        Preconditions.checkArgument(login != null ^ uid != null,
                "Можно задать либо логин, либо uid");
        this.login = login;
        this.uid = uid;
    }

    public static LoginOrUid of(@Nullable String login, @Nullable Long uid) {
        return new LoginOrUid(login, uid);
    }

    public static LoginOrUid of(@Nonnull String login) {
        return new LoginOrUid(login, null);
    }

    public static LoginOrUid of(@Nonnull Long uid) {
        return new LoginOrUid(null, uid);
    }

    @Nonnull
    @SuppressWarnings("ConstantConditions")
    public Object get() {
        return login != null ? login : uid;
    }

    public boolean isLogin() {
        return login != null;
    }

    public boolean isUid() {
        return uid != null;
    }

    @Nonnull
    public String getLogin() throws NullPointerException {
        return Objects.requireNonNull(login, "Задан uid, а не login");
    }

    @Nonnull
    public Long getUid() throws NullPointerException {
        return Objects.requireNonNull(uid, "Задан login, а не uid");
    }

    /**
     * Применяет метод в зависимости от того, какое из значений передано.
     * Полезно когда есть два overloaded-метода, зависящих от входного параметра.
     */
    @Nullable
    public <R> R map(Function<String, R> leftFunc, Function<Long, R> rightFunc) {
        return login != null ? leftFunc.apply(login) : rightFunc.apply(uid);
    }
}
