package ru.yandex.direct.dbutil.model;

import java.util.Objects;

import javax.annotation.Nonnull;

/**
 * Тройка Uid + ClientId + shard клиента
 */
public final class UidClientIdShard {
    /**
     * Идентификатор пользователя в паспорте
     */
    private final Long uid;
    /**
     * Идентификатор пользователя в балансе
     */
    private final ClientId clientId;
    /**
     * shard на котором расположены данные клиента
     */
    private final int shard;

    private UidClientIdShard(Long uid, ClientId clientId, int shard) {
        this.uid = Objects.requireNonNull(uid, "uid");
        this.clientId = Objects.requireNonNull(clientId, "clientId");
        this.shard = shard;
    }

    @Nonnull
    public static UidClientIdShard of(Long uid, Long clientId, int shard) {
        return of(uid, ClientId.fromLong(clientId), shard);
    }

    @Nonnull
    public static UidClientIdShard of(Long uid, ClientId clientId, int shard) {
        return new UidClientIdShard(uid, clientId, shard);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        UidClientIdShard that = (UidClientIdShard) o;

        if (shard != that.shard) {
            return false;
        }
        if (!uid.equals(that.uid)) {
            return false;
        }
        return clientId.equals(that.clientId);
    }

    @Override
    public int hashCode() {
        int result = uid.hashCode();
        result = 31 * result + clientId.hashCode();
        result = 31 * result + shard;
        return result;
    }

    @Nonnull
    public Long getUid() {
        return uid;
    }

    @Nonnull
    public ClientId getClientId() {
        return clientId;
    }

    public int getShard() {
        return shard;
    }

    public UidAndClientId getUidAndClientId() {
        return UidAndClientId.of(getUid(), getClientId());
    }
}
