package ru.yandex.direct.rbac;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;

import com.google.common.collect.Sets;

import static java.util.stream.Collectors.joining;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

public enum ClientPerm {
    // CHECKSTYLE:OFF
    SUPER_SUBCLIENT("super_subclient"),
    XLS_IMPORT("xls_import"),
    MASS_ADVQ("mass_advq"),
    MONEY_TRANSFER("money_transfer"),
    INTERNAL_AD_PRODUCT("internal_ad_product"),
    ;
    // CHECKSTYLE:ON

    private static final Map<String, ClientPerm> PERMS_BY_NAME =
            listToMap(Arrays.asList(ClientPerm.values()), p -> p.dbName, Function.identity());
    private final String dbName;

    ClientPerm(String dbName) {
        this.dbName = dbName;
    }

    private String dbName() {
        return dbName;
    }

    public static Set<ClientPerm> parse(String str) {
        if (str.isEmpty()) {
            return EnumSet.noneOf(ClientPerm.class);
        }
        return Arrays.stream(str.split(","))
                .map(s -> Optional.ofNullable(PERMS_BY_NAME.get(s)))
                .map(s -> s.<IllegalArgumentException>orElseThrow(() ->
                        new IllegalArgumentException("Unknown perm: " + s)))
                .collect(Sets.toImmutableEnumSet());
    }

    @Nonnull
    public static String format(@Nonnull Set<ClientPerm> perms) {
        if (perms.isEmpty()) {
            return "";
        }
        return perms.stream().map(ClientPerm::dbName)
                .collect(joining(","));
    }

}
