package ru.yandex.direct.rbac;

import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import ru.yandex.direct.dbschema.ppc.enums.ClientsRole;

/**
 * Роли rbac
 */
public enum RbacRole {
    SUPER(true, ClientsRole.super_),
    SUPERREADER(true, ClientsRole.superreader),
    SUPPORT(true, ClientsRole.support),
    PLACER(true, ClientsRole.placer),
    MEDIA(true, ClientsRole.media),
    MANAGER(true, ClientsRole.manager),
    INTERNAL_AD_ADMIN(true, ClientsRole.internal_ad_admin),
    INTERNAL_AD_MANAGER(true, ClientsRole.internal_ad_manager),
    INTERNAL_AD_SUPERREADER(true, ClientsRole.internal_ad_superreader),

    AGENCY(false, ClientsRole.agency),
    CLIENT(false, ClientsRole.client),
    LIMITED_SUPPORT(false, ClientsRole.limited_support),
    EMPTY(false, ClientsRole.empty);

    private final boolean isInternal;
    private final ClientsRole sourceRole;

    private static final Map<ClientsRole, RbacRole> FROM_SOURCE_MAP = Stream.of(RbacRole.values())
            .collect(Collectors.toMap(role -> role.sourceRole, Function.identity()));

    RbacRole(boolean isInternal, ClientsRole sourceRole) {
        this.isInternal = isInternal;
        this.sourceRole = sourceRole;
    }

    /**
     * Check if role is "internal" (for Yandex staff only)
     *
     * @return true, if role is internal
     */
    public boolean isInternal() {
        return isInternal;
    }

    /**
     * Check if role is one of specified
     *
     * @param roles - array of roles to check
     * @return - true, if current role is in roles list
     */
    public boolean anyOf(RbacRole... roles) {
        for (RbacRole role : roles) {
            if (this == role) {
                return true;
            }
        }
        return false;
    }

    /**
     * Сконвертировать роль из ppc.clients в RbacROle
     */
    public static RbacRole fromSource(ClientsRole ppcRole) {
        if (ppcRole == null) {
            return null;
        }

        if (!FROM_SOURCE_MAP.containsKey(ppcRole)) {
            throw new IllegalArgumentException("Unknown role: " + ppcRole);
        }

        return FROM_SOURCE_MAP.get(ppcRole);
    }


    public static ClientsRole toSource(RbacRole role) {
        if (role == null) {
            return null;
        }

        return role.sourceRole;
    }
}
