package ru.yandex.direct.bmapi.client

import io.netty.handler.codec.http.HttpHeaders
import org.slf4j.LoggerFactory
import org.springframework.http.MediaType
import ru.yandex.direct.asynchttp.ParallelFetcher
import ru.yandex.direct.asynchttp.ParallelFetcherFactory
import ru.yandex.direct.asynchttp.Result
import ru.yandex.direct.bmapi.api.BmApi
import ru.yandex.direct.bmapi.client.model.BmApiFeedInfoResponse
import ru.yandex.direct.bmapi.client.model.BmApiRequest
import ru.yandex.direct.bmapi.configuration.BmapiConfiguration
import ru.yandex.direct.core.entity.feed.model.BusinessType
import ru.yandex.direct.core.entity.feed.model.UpdateStatus
import ru.yandex.direct.http.smart.core.Smart
import java.time.Duration
import kotlin.math.roundToLong

private val LOGGER = LoggerFactory.getLogger(BmapiClient::class.java)

open class BmapiClient(
    configuration: BmapiConfiguration,
    private val parallelFetcherFactory: ParallelFetcherFactory
) {

    companion object {
        // Максимальный размер фида в байтах по умолчанию, для некоторых клиентов может быть переопределен,
        // в таких случаях надо увеличивать таймаут соответсвенно
        const val DEFAULT_MAX_FEED_SIZE = 512 * 1024 * 1024 //TODO: move to configuration
    }

    private val api: BmApi

    open fun processFeeds(bmapiRequests: List<BmApiRequest>): Map<Long, Result<BmApiFeedInfoResponse>> {
        val calls = bmapiRequests.map {
            api.call(
                it.feedId,
                "yml2directinf",
                it.url,
                BusinessType.toSource(it.businessType)!!.literal,
                it.lastValidFeedType?.typedValue,
                it.maxFileSizeInBytes,
                "bytes",
                UpdateStatus.toSource(it.updateStatus)?.literal,
                it.feedId,
                1,
                it.login,
                it.password
            ).request
        }

        LOGGER.info("Sending to BmAPI.yml2directinf feeds with id ${bmapiRequests.map { it.feedId }}")
        var parallelFetcher: ParallelFetcher<BmApiFeedInfoResponse> =
            parallelFetcherFactory.getParallelFetcher()

        if (bmapiRequests.any { it.maxFileSizeInBytes > DEFAULT_MAX_FEED_SIZE }) {
            val timeoutCoef = bmapiRequests.maxOf { it.maxFileSizeInBytes }.toDouble().div(DEFAULT_MAX_FEED_SIZE)
            val fetcherSetting = parallelFetcherFactory.defaultSettingsCopy()
            val requestFetcherSetting =
                fetcherSetting.withRequestTimeout(
                    Duration.ofMillis((fetcherSetting.requestTimeout.toMillis() * timeoutCoef).roundToLong())
                )
            parallelFetcher = parallelFetcherFactory.getParallelFetcher(requestFetcherSetting)
        }

        return parallelFetcher.execute(calls)
    }

    init {
        this.api = Smart.builder()
            .withParallelFetcherFactory(parallelFetcherFactory)
            .withProfileName("bmapi_client")
            .withBaseUrl(configuration.apiUrl)
            .addHeaderConfigurator { headers: HttpHeaders ->
                headers.add(org.springframework.http.HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
            }
            .build()
            .create(BmApi::class.java)
    }
}
