package ru.yandex.direct.bs.dspcreative.component;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.adv.direct.dspcreative.TDspCreative;
import ru.yandex.direct.bs.dspcreative.exception.DspCreativeConverterException;
import ru.yandex.direct.bs.dspcreative.model.DspCreativeConverterType;
import ru.yandex.direct.bs.dspcreative.model.DspCreativeExportEntry;
import ru.yandex.direct.bs.dspcreative.model.DspCreativeParameterValue;
import ru.yandex.direct.utils.HashingUtils;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.JsonUtils.toDeterministicJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;

/*
 * Базовый класс для конвертации креативов к пригодному к выгрузке в YT виду (TDspCreative).
 * После переезда bannerstorage в аркадию надо вынести данный класс в общую библиотеку
 */
@ParametersAreNonnullByDefault
public abstract class DspCreativeBaseConverter {
    private static final Logger logger = LoggerFactory.getLogger(DspCreativeBaseConverter.class);

    private static final Set<Long> YABS_DSP_IDS = Set.of(1L);
    private static final String HAS_OFFER_NAME_PARAM_NAME = "HAS_OFFER_NAME";

    private final DspCreativeUtils dspCreativeUtils;

    public DspCreativeBaseConverter(DspCreativeUtils dspCreativeUtils) {
        this.dspCreativeUtils = dspCreativeUtils;
    }

    abstract DspCreativeConverterType getConverterType();

    abstract String defaultStaticData(DspCreativeExportEntry dspCreativeExportEntry);

    /*
    Преобразовать DspCreativeExportEntry к TDspCreative (для выгрузки в YT)
     */
    public TDspCreative convert(DspCreativeExportEntry dspCreativeEntry) {
        logger.info("Start converting DspCreativeExportEntry: {}", toJson(dspCreativeEntry));

        validate(dspCreativeEntry);

        String defaultStaticData = defaultStaticData(dspCreativeEntry);
        String customObjectId = extractCustomObjectId(dspCreativeEntry, defaultStaticData);
        String staticDataJson = extractStaticDataJson(dspCreativeEntry, defaultStaticData, customObjectId);

        TDspCreative dspCreative = TDspCreative.newBuilder()
                .setCreativeID(dspCreativeEntry.getCreativeId())
                .setIterID(dspCreativeUtils.generateDspCreativeIterId())
                .setDSPID(dspCreativeEntry.getDspId())
                .setTag(nvl(dspCreativeEntry.getTag(), ""))
                .setCreativeTemplateID(nvl(dspCreativeEntry.getCreativeTemplateId(), 0L))
                .setCreativeCodeID(nvl(dspCreativeEntry.getCreativeCodeId(), 0L))
                .setCreateTime(dspCreativeUtils.convertDateToTimestamp(dspCreativeEntry.getCreateTime()))
                .setUpdateTime(dspCreativeUtils.getCurrentTimestamp())
                .setExpireTime(dspCreativeUtils.convertDateToTimestamp(dspCreativeEntry.getExpireTime()))
                .setCreativeVersionID(dspCreativeEntry.getCreativeVersionId())
                .addAllTNSBrand(nvl(dspCreativeEntry.getTnsBrand(), Collections.emptyList()))
                .addAllTNSArticle(nvl(dspCreativeEntry.getTnsArticle(), Collections.emptyList()))
                .addAllGeo(convertGeo(dspCreativeEntry))
                .addAllSite(nvl(dspCreativeEntry.getSite(), Collections.emptyList()))
                .addAllDomainList(convertDomainList(dspCreativeEntry))
                .setOptions(createOptionsMask(dspCreativeEntry))
                .setPreModeratedObjects(customObjectId)
                .setStaticDataJson(nvl(staticDataJson, ""))
                .setData(nvl(dspCreativeEntry.getData(), ""))
                .setWidth(dspCreativeEntry.getWidth())
                .setHeight(dspCreativeEntry.getHeight())
                .setMaxObjLimit(nvl(dspCreativeEntry.getMaxShowItems(), 0L))
                .setMinObjLimit(nvl(dspCreativeEntry.getMinShowItems(), 0L))
                .setFirstShowObjCount(nvl(dspCreativeEntry.getFirstShowItems(), 0L))
                .setParamValuesJson(convertParamValuesToJson(dspCreativeEntry.getParameterValues()))
                .setToken(dspCreativeEntry.getToken() != null ?
                        HashingUtils.getYabsMd5HalfHashUtf8(dspCreativeEntry.getToken()).longValue() : 0L)
                .setConstructorDataJson(nvl(dspCreativeEntry.getConstructorData(), ""))
                .setSmartThemeID(nvl(dspCreativeEntry.getSmartThemeId(), 0L))
                .setSmartLayoutID(nvl(dspCreativeEntry.getSmartLayoutId(), 0L))
                .setSmartSizeID(nvl(dspCreativeEntry.getSmartSizeId(), 0L))
                .build();

        dspCreativeUtils.addDomainListToTargetDomain(dspCreative.getDomainListList());

        logger.info("Successfully converted");

        return dspCreative;
    }

    private List<Long> convertGeo(DspCreativeExportEntry dspCreativeEntry) {
        if (YABS_DSP_IDS.contains(dspCreativeEntry.getDspId())) {
            return Collections.emptyList();
        }
        return nvl(dspCreativeEntry.getGeo(), Collections.emptyList());
    }

    private List<String> convertDomainList(DspCreativeExportEntry dspCreativeEntry) {
        if (YABS_DSP_IDS.contains(dspCreativeEntry.getDspId())) {
            return Collections.emptyList();
        }
        return nvl(dspCreativeEntry.getDomainList(), Collections.emptyList());
    }

    /*
    Валидируем запись на правильность заполнения полей
     */
    private void validate(DspCreativeExportEntry dspCreativeEntry) {
        checkRequiredFields(dspCreativeEntry);
        checkJsonFields(dspCreativeEntry);
        checkNotEmptyStringFields(dspCreativeEntry);
    }

    /*
    Проверяем, что в json-полях лежит валидный json
     */
    private void checkJsonFields(DspCreativeExportEntry dspCreativeEntry) {
        checkJsonFieldValue(dspCreativeEntry.getStaticData(), "StaticData");
        checkJsonFieldValue(dspCreativeEntry.getConstructorData(), "ConstructorData");
    }

    /*
    Проверяем значение поля на json-валидность
     */
    private void checkJsonFieldValue(@Nullable String value, String fieldName) {
        if (value == null || value.isEmpty()) {
            return;
        }
        try {
            new ObjectMapper().readTree(value);
        } catch (IOException e) {
            throw new DspCreativeConverterException(String.format("Can't parse json field %s", fieldName));
        }
    }

    /*
    Проверяем, что в полях не пустые строки
     */
    private void checkNotEmptyStringFields(DspCreativeExportEntry dspCreativeEntry) {
        if (dspCreativeEntry.getIsGeoPin() == null || !dspCreativeEntry.getIsGeoPin()) {
            checkNotEmptyStringFieldValue(dspCreativeEntry.getData(), "Data");
        }
    }

    /*
    Проверяем значение поля на пустоту
     */
    private void checkNotEmptyStringFieldValue(@Nullable String value, String fieldName) {
        if (value == null || value.isEmpty()) {
            throw new DspCreativeConverterException(String.format("Invalid field %s value", fieldName));
        }
    }

    /*
    Проверяем, что обязательные поля не null
     */
    private void checkRequiredFields(DspCreativeExportEntry dspCreativeEntry) {
        try {
            checkCommonRequiredFields(dspCreativeEntry);
            checkCustomRequiredFields(dspCreativeEntry);
        } catch (NullPointerException e) {
            throw new DspCreativeConverterException("Some required fields are missing", e);
        }
    }

    /*
    Проверяем общие обязательные поля
     */
    private void checkCommonRequiredFields(DspCreativeExportEntry dspCreativeEntry) {
        Preconditions.checkNotNull(dspCreativeEntry.getCreativeId(), "CreativeId is required");
        Preconditions.checkNotNull(dspCreativeEntry.getDspId(), "DspId is required");
        Preconditions.checkNotNull(dspCreativeEntry.getCreativeVersionId(), "CreativeVersionId is required");
        Preconditions.checkNotNull(dspCreativeEntry.getData(), "Data is required");
        Preconditions.checkNotNull(dspCreativeEntry.getIsEnabled(), "IsEnabled is required");
        Preconditions.checkNotNull(dspCreativeEntry.getIsVideo(), "IsVideo is required");
        Preconditions.checkNotNull(dspCreativeEntry.getWidth(), "Width is required");
        Preconditions.checkNotNull(dspCreativeEntry.getHeight(), "Height is required");
    }

    /*
    Проверяем наличие обязательных полей
     */
    abstract void checkCustomRequiredFields(DspCreativeExportEntry dspCreativeEntry);

    /*
    Создаем из отдельных опций их общую маску
     */
    private long createOptionsMask(DspCreativeExportEntry dspCreativeEntry) {
        long mask = 0;
        if (dspCreativeEntry.getIsEnabled() != null && !dspCreativeEntry.getIsEnabled()) {
            mask = mask | (1L << TDspCreative.DspCreativeOptionsMask.DECLINED_VALUE);
        }
        if (dspCreativeEntry.getIsVideo() != null && dspCreativeEntry.getIsVideo()) {
            mask = mask | (1L << TDspCreative.DspCreativeOptionsMask.VIDEO_VALUE);
        }
        if (dspCreativeEntry.getPostmoderated() == null || dspCreativeEntry.getPostmoderated()) {
            mask = mask | (1L << TDspCreative.DspCreativeOptionsMask.POSTMODERATED_VALUE);
        }
        if (dspCreativeEntry.getIsStatic() != null && dspCreativeEntry.getIsStatic()) {
            mask = mask | (1L << TDspCreative.DspCreativeOptionsMask.STATIC_VALUE);
        }
        if (dspCreativeEntry.getIsAdaptive() != null && dspCreativeEntry.getIsAdaptive()) {
            mask = mask | (1L << TDspCreative.DspCreativeOptionsMask.ADAPTIVE_VALUE);
        }
        if (dspCreativeEntry.getIsAudio() != null && dspCreativeEntry.getIsAudio()) {
            mask = mask | (1L << TDspCreative.DspCreativeOptionsMask.AUDIO_VALUE);
        }
        if (dspCreativeEntry.getIsGeoPin() != null && dspCreativeEntry.getIsGeoPin()) {
            mask = mask | (1L << TDspCreative.DspCreativeOptionsMask.GEO_PIN_VALUE);
        }
        if (YABS_DSP_IDS.contains(dspCreativeEntry.getDspId()) && (
                dspCreativeEntry.getIsVideo() != null && dspCreativeEntry.getIsVideo() ||
                dspCreativeEntry.getIsAudio() != null && dspCreativeEntry.getIsAudio() ||
                dspCreativeEntry.getIsGeoPin() != null && dspCreativeEntry.getIsGeoPin())) {
            mask = mask | (1L << TDspCreative.DspCreativeOptionsMask.NO_OBJECT_LOG_VALUE);
        }
        if (findHasOfferName(dspCreativeEntry.getParameterValues())) {
            mask = mask | (1L << TDspCreative.DspCreativeOptionsMask.HAS_OFFER_NAME_VALUE);
        }
        return mask;
    }

    /*
    Находим в ParamValues параметр HasOfferName
     */
    private boolean findHasOfferName(List<DspCreativeParameterValue> parameterValues) {
        return parameterValues
                .stream()
                .noneMatch(param -> HAS_OFFER_NAME_PARAM_NAME.equals(param.getParamName()) &&
                        param.getParamValues() != null &&
                        !param.getParamValues().isEmpty() &&
                        "0".equals(param.getParamValues().get(0)));
    }

    /*
    Конвертируем ParamValues в json
     */
    private String convertParamValuesToJson(List<DspCreativeParameterValue> parameterValues) {
        // для одинаковых ключей берем последнее значение
        Map<String, List<String>> parameterValuesMap = parameterValues
                .stream()
                .collect(Collectors.toMap(
                        DspCreativeParameterValue::getParamName,
                        DspCreativeParameterValue::getParamValues,
                        (l, r) -> r));

        return toDeterministicJson(parameterValuesMap);
    }

    /*
    Достаем CustomObjectId
     */
    private String extractCustomObjectId(DspCreativeExportEntry dspCreativeEntry, @Nullable String staticData) {
        if (YABS_DSP_IDS.contains(dspCreativeEntry.getDspId()) && staticData != null &&
                !staticData.isEmpty()) {
            return "C" + dspCreativeEntry.getCreativeId();
        }
        return "";
    }

    /*
    Достаем StaticData с особыми преобразованиями
     */
    private String extractStaticDataJson(DspCreativeExportEntry dspCreativeEntry,
                                         @Nullable String staticData,
                                         String customObjectId) {
        if (staticData == null || staticData.isEmpty()) {
            return staticData;
        }

        final JsonNode staticDataJson;
        try {
            staticDataJson = new ObjectMapper().readTree(staticData);
        } catch (IOException e) {
            throw new DspCreativeConverterException("Unexpected error", e);
        }

        if (YABS_DSP_IDS.contains(dspCreativeEntry.getDspId()) && !customObjectId.isEmpty()) {
            if (staticDataJson.has("logo")) {
                ((ObjectNode) staticDataJson.get("logo")).put("object_id", customObjectId);
            }
            if (staticDataJson.has("spec")) {
                ((ObjectNode) staticDataJson.get("spec")).put("object_id", customObjectId);
            }
        }
        ((ObjectNode) staticDataJson).put("creative_id", dspCreativeEntry.getCreativeId().toString());
        return toDeterministicJson(staticDataJson);
    }
}
