package ru.yandex.direct.bs.id.generator;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

import org.asynchttpclient.AsyncHttpClient;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;

import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;

/**
 * Клиент для генерации id домента в БК
 */
public class BsDomainIdGeneratorClient {

    private final BsDomainIdGeneratorApi api;

    public BsDomainIdGeneratorClient(BsDomainIdGeneratorClientConfig clientConfig, AsyncHttpClient asyncHttpClient,
                                     TvmIntegration tvmIntegration) {
        var fetcherSettings = new FetcherSettings()
                .withConnectTimeout(clientConfig.getConnectTimeout())
                .withRequestTimeout(clientConfig.getRequestTimeout())
                .withRequestRetries(clientConfig.getRequestRetries());

        var parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient, fetcherSettings);
        var tvmService = TvmService.fromId(clientConfig.getTvmId());
        this.api = Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withProfileName("generate_bs_domain_id_api")
                .withBaseUrl(clientConfig.getUrl())
                .useTvm(tvmIntegration, tvmService)
                .build()
                .create(BsDomainIdGeneratorApi.class);
    }

    /**
     * На вход передается спиок доменов, для которых нужно получить id
     * Чанкования внутри метода нет
     * Возвращает словарь domain -> id
     * Если БК вернуло id'шников меньше, чем доменов, будет брошен {@link BsDomainIdGeneratorException}
     */
    public Map<String, Long> generate(Collection<String> domains) {
        var distinctDomains = domains.stream().distinct().collect(Collectors.toUnmodifiableList());
        var request = new BsDomainIdGeneratorRequest(distinctDomains);
        Call<BsDomainIdGeneratorResponse> responseCall = api.generate(request);
        var result = responseCall.execute();
        checkResultForErrors(result, BsDomainIdGeneratorException::new);
        var ids = result.getSuccess().getIds();
        if (distinctDomains.size() != ids.size()) {
            throw new BsDomainIdGeneratorException(
                    String.format("Response size not equals to request size %d != %d.%nRequest: %s%nResponse: %s",
                            distinctDomains.size(), ids.size(), distinctDomains, ids));
        }
        Map<String, Long> domainsIdsMap = new HashMap<>();
        for (int i = 0; i < distinctDomains.size(); i++) {
            domainsIdsMap.put(distinctDomains.get(i), ids.get(i));
        }
        return domainsIdsMap;
    }
}
