package ru.yandex.direct.bstransport.yt.repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import com.google.protobuf.Message;

import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.misc.concurrent.TimeoutRuntimeException;
import ru.yandex.misc.thread.ExecutionRuntimeException;
import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

/**
 * Базовый репозиторий для выгрузки данных в БК
 * Записывает данные в "Единую базу" и в очередь Caesar
 *
 * @param <T> protobuf сообщения в Caesar
 */

public abstract class BaseBsExportYtRepository<T extends Message> {
    private final YtProvider ytProvider;
    private final YtCluster ytCluster;
    private final String tablePath;
    private final String queuePath;
    private final long transactionTimeoutSec;
    protected final CaesarQueueRequestFactory caesarQueueRequestFactory;
    private RequestSettings<T> modifyRequestSettings;
    private RequestSettings<T> deleteRequestSettings;

    private static final Integer CHUNK_SIZE = 50_000;

    public BaseBsExportYtRepository(BsExportYtRepositoryContext context, String tablePath) {
        this(context, tablePath, null);
    }

    public BaseBsExportYtRepository(BsExportYtRepositoryContext context, String tablePath, String queuePath) {
        this.ytProvider = context.getYtProvider();
        var ytConfig = context.getYtConfig();
        this.ytCluster = ytConfig.getCluster();
        this.tablePath = tablePath;
        this.queuePath = queuePath;
        this.transactionTimeoutSec = ytConfig.getTransactionTimeout().getSeconds();
        this.caesarQueueRequestFactory = context.getCaesarQueueWriter();
        init();
    }

    /**
     * Маппинг protobuf для Caesar в схему таблицы "Единой базы"
     */
    public abstract List<ColumnMapping<T>> getSchemaWithMapping();

    private void init() {
        List<ColumnMapping<T>> mappers = new ArrayList<>(getSchemaWithMapping());

        var tableSchemaBuilder = new TableSchema.Builder();
        mappers.forEach(schemaWithMapping ->
                tableSchemaBuilder.add(schemaWithMapping.getColumnSchema()));

        List<ColumnMapping<T>> deleteMappers = mappers.stream()
                .filter(schemaWithMapping -> Objects.nonNull(schemaWithMapping.getColumnSchema().getSortOrder()))
                .collect(Collectors.toList());
        var tableSchema = tableSchemaBuilder.build();

        this.modifyRequestSettings = new RequestSettings<T>()
                .setMappers(mappers)
                .setSchema(tableSchema)
                .setRequestOperation(ModifyRowsRequest::addInsert);

        this.deleteRequestSettings = new RequestSettings<T>()
                .setMappers(deleteMappers)
                .setSchema(tableSchema.toKeys())
                .setRequestOperation(ModifyRowsRequest::addDelete);
    }

    /**
     * Вставить или изменить объекты в базе и отправить изменение в очередь
     */
    public void modify(Collection<T> rows) {
        doRequest(rows, modifyRequestSettings);
    }

    /**
     * Удалить объект из базы и отправить удаление в очередь
     */
    public void delete(Collection<T> rows) {
        doRequest(rows, deleteRequestSettings);
    }

    private void doRequest(Collection<T> rows, RequestSettings<T> requestsSettings) {
        for (var rowsChunk : Iterables.partition(rows, CHUNK_SIZE)) {
            ModifyRowsRequest request = new ModifyRowsRequest(tablePath, requestsSettings.schema);
            request.setRequireSyncReplica(false);
            ModifyRowsRequest queueRequest
                    = ifNotNull(queuePath, q -> caesarQueueRequestFactory.getRequest(rowsChunk, q));
            for (var row : rowsChunk) {
                Map<String, Object> result = new HashMap<>();
                requestsSettings.mappers.forEach(settings -> result.put(settings.getName(),
                        settings.extractValue(row)));
                requestsSettings.requestOperation.accept(request, result);
            }
            doTransactionRequest(request, queueRequest);
        }
    }

    protected void doTransactionRequest(ModifyRowsRequest tableModifyRequest, ModifyRowsRequest queueModifyRequest) {
        ytProvider.getDynamicOperator(ytCluster)
                .runInTransaction(tr -> sendRequest(tr, tableModifyRequest, queueModifyRequest),
                        new ApiServiceTransactionOptions(ETransactionType.TT_TABLET).setSticky(true));
    }

    private void sendRequest(ApiServiceTransaction transaction, ModifyRowsRequest tableModifyRequest,
                             ModifyRowsRequest queueModifyRequest) {
        CompletableFuture<Void> future;
        if (Objects.nonNull(tableModifyRequest)) {
            future = transaction.modifyRows(tableModifyRequest);
        } else {
            future = CompletableFuture.completedFuture(null);
        }
        if (Objects.nonNull(queueModifyRequest)) {
            future = future.thenCompose(unused -> transaction.modifyRows(queueModifyRequest));
        }
        try {
            future.get(transactionTimeoutSec, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        } catch (ExecutionException e) {
            throw new ExecutionRuntimeException(e);
        } catch (TimeoutException e) {
            throw new TimeoutRuntimeException(e);
        }
    }

    protected String getTablePath() {
        return tablePath;
    }

    protected String getQueuePath() {
        return queuePath;
    }

    private static class RequestSettings<T> {
        private BiConsumer<ModifyRowsRequest, Map<String, Object>> requestOperation;
        private List<ColumnMapping<T>> mappers;
        private TableSchema schema;

        public RequestSettings<T> setRequestOperation(BiConsumer<ModifyRowsRequest, Map<String, Object>> requestOperation) {
            this.requestOperation = requestOperation;
            return this;
        }

        public RequestSettings<T> setMappers(List<ColumnMapping<T>> mappers) {
            this.mappers = mappers;
            return this;
        }

        public RequestSettings<T> setSchema(TableSchema schema) {
            this.schema = schema;
            return this;
        }
    }
}
