package ru.yandex.direct.bstransport.yt.repository

import com.google.protobuf.Message
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeProtoUtils
import ru.yandex.inside.yt.kosher.ytree.YTreeNode
import ru.yandex.yt.ytclient.tables.ColumnSchema
import ru.yandex.yt.ytclient.tables.ColumnSortOrder
import ru.yandex.yt.ytclient.tables.ColumnValueType

class ColumnDescription<T>(val columnSchema: ColumnSchema, private val conv: (YTreeNode?) -> T?) {
    fun getValue(node: YTreeNode?): T? = conv(node)
    val name: String
        get() = columnSchema.name

    val type: ColumnValueType
        get() = columnSchema.type
}

fun int64Column(name: String, sortOrder: ColumnSortOrder? = null): ColumnDescription<Long> {
    val columnSchema = ColumnSchema(name, ColumnValueType.INT64, sortOrder)
    return ColumnDescription(columnSchema) { it?.longValue() ?: 0L }
}


fun uint64Column(name: String, sortOrder: ColumnSortOrder? = null): ColumnDescription<Long> {
    val columnSchema = ColumnSchema(name, ColumnValueType.UINT64, sortOrder)
    return ColumnDescription(columnSchema) { it?.longValue() ?: 0L }
}


fun stringColumn(name: String, sortOrder: ColumnSortOrder? = null): ColumnDescription<String> {
    val columnSchema = ColumnSchema(name, ColumnValueType.STRING, sortOrder)
    return ColumnDescription(columnSchema) { it?.stringValue() ?: "" }
}

fun boolColumn(name: String, sortOrder: ColumnSortOrder? = null): ColumnDescription<Boolean> {
    val columnSchema = ColumnSchema(name, ColumnValueType.BOOLEAN, sortOrder)
    return ColumnDescription(columnSchema) { it?.boolValue() ?: false }
}

fun doubleColumn(name: String, sortOrder: ColumnSortOrder? = null): ColumnDescription<Double> {
    val columnSchema = ColumnSchema(name, ColumnValueType.DOUBLE, sortOrder)
    return ColumnDescription(columnSchema) { it?.doubleValue() ?: 0.0 }
}

fun <T : Message> protoColumn(name: String, clazz: Class<T>, sortOrder: ColumnSortOrder? = null): ColumnDescription<T> {
    val columnSchema = ColumnSchema(name, ColumnValueType.ANY, sortOrder)
    return ColumnDescription(columnSchema) { if (it != null) YTreeProtoUtils.unmarshal(it, clazz) else null }
}

fun customColumn(name: String, sortOrder: ColumnSortOrder? = null): ColumnDescription<YTreeNode> {
    val columnSchema = ColumnSchema(name, ColumnValueType.ANY, sortOrder)
    return ColumnDescription(columnSchema) { it }
}
