package ru.yandex.direct.bstransport.yt.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.springframework.stereotype.Component;

import ru.yandex.adv.direct.expression.MultiplierChangeRequest;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeProtoUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeListNode;
import ru.yandex.yt.ytclient.tables.ColumnSchema;
import ru.yandex.yt.ytclient.tables.ColumnValueType;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.bstransport.yt.repository.ColumnMapping.column;
import static ru.yandex.yt.ytclient.tables.ColumnSortOrder.ASCENDING;

@Component
public class MultipliersYtRepository extends BaseBsExportYtRepository<MultiplierChangeRequest> {
    public static final String ORDER_ID = "OrderID";
    public static final String AD_GROUP_ID = "AdGroupID";
    public static final String TYPE = "Type";

    public MultipliersYtRepository(BsExportYtRepositoryContext context) {
        super(context, context.getYtConfig().getMultipliersTable(), context.getYtConfig().getMultipliersQueue());
    }

    @Override
    public List<ColumnMapping<MultiplierChangeRequest>> getSchemaWithMapping() {
        return List.of(
                column(new ColumnSchema(ORDER_ID, ColumnValueType.INT64, ASCENDING), this::getOrderId),
                column(new ColumnSchema(AD_GROUP_ID, ColumnValueType.INT64, ASCENDING), this::getAdGroupId),
                column(new ColumnSchema(TYPE, ColumnValueType.STRING, ASCENDING), this::getType),
                column(new ColumnSchema("IsEnabled", ColumnValueType.BOOLEAN), this::getIsEnabled),
                column(new ColumnSchema("Multipliers", ColumnValueType.ANY), this::getMultipliers)
        );
    }

    private Long getOrderId(MultiplierChangeRequest r) {
        return r.hasUpsertRequest() ? r.getUpsertRequest().getOrderID() : r.getDeleteRequest().getOrderID();
    }

    private Long getAdGroupId(MultiplierChangeRequest r) {
        return r.hasUpsertRequest() ? r.getUpsertRequest().getAdGroupID() : r.getDeleteRequest().getAdGroupID();
    }

    private String getType(MultiplierChangeRequest r) {
        var type = r.hasUpsertRequest() ? r.getUpsertRequest().getType() : r.getDeleteRequest().getType();
        return type.toString();
    }

    private Boolean getIsEnabled(MultiplierChangeRequest r) {
        checkIfUpsertRequestExists(r);
        return r.getUpsertRequest().getIsEnabled();
    }

    private YTreeListNode getMultipliers(MultiplierChangeRequest r) {
        checkIfUpsertRequestExists(r);
        YTreeBuilder multipliersList = YTree.builder().beginList();
        r.getUpsertRequest().getMultipliersList().stream()
                .map(YTreeProtoUtils::marshal)
                .forEach(multipliersList::value);
        return multipliersList.buildList();
    }

    private void checkIfUpsertRequestExists(MultiplierChangeRequest r) {
        checkArgument(r.hasUpsertRequest(), "Modified request must contains UpsertRequest");
    }

    public void changeMultipliers(Collection<MultiplierChangeRequest> requests) {
        Map<Boolean, List<MultiplierChangeRequest>> upsertedAndDeleted =
                requests.stream().collect(Collectors.groupingBy(MultiplierChangeRequest::hasUpsertRequest));
        if (upsertedAndDeleted.containsKey(Boolean.FALSE)) {
            delete(upsertedAndDeleted.get(Boolean.FALSE));
        }
        // важно, чтобы модификация шла после удаления, т.к. модификация оперирует
        // более актуальными данными нежели удаление
        if (upsertedAndDeleted.containsKey(Boolean.TRUE)) {
            modify(upsertedAndDeleted.get(Boolean.TRUE));
        }
    }
}
