package ru.yandex.direct.bstransport.yt.repository.adgroup

import com.google.protobuf.Message
import org.slf4j.LoggerFactory
import ru.yandex.direct.bstransport.yt.repository.BaseBsExportYtRepository
import ru.yandex.direct.bstransport.yt.repository.BsExportYtRepositoryContext
import ru.yandex.direct.bstransport.yt.repository.common.RowPosition
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode
import java.time.Instant

data class AdGroupsPosition(
    val ytHash: Long,
    val adGroupsId: Long,
): RowPosition()

data class AdGroupWithYtHash<T : Message>(
    val adGroup: T,
    val ytHash: Long,
)

abstract class AdGroupCommonYtRepository<T : Message>(
    private val context: BsExportYtRepositoryContext,
    tablePath: String,
    queuePath: String
) : BaseBsExportYtRepository<T>(
    context, tablePath, queuePath
) {
    companion object {
        private val logger = LoggerFactory.getLogger(AdGroupCommonYtRepository::class.java)
        const val WRITE_CHUNK_SIZE = 100_000
        private val ytHashExpressionRegex = "uint64\\(farm_hash\\(AdGroupID\\)%(\\d+)\\)".toRegex(RegexOption.IGNORE_CASE)
    }

    abstract fun buildAdGroupFromYtRow(node: YTreeMapNode, updateTime: Long): T

    /**
     * Читает из сортированной таблицы протобуф с ресурсами
     * Выбирает те, у которых AdGroupID > param.adGroupID
     * В поле UpdateTime записывает текущее время
     */
    fun getAdGroupsForResync(position: AdGroupsPosition, limit: Int, maxYtHash: Long? = null): List<AdGroupWithYtHash<T>> {

        val startTimeStamp = Instant.now().epochSecond
        val query = """
            * FROM [$tablePath]
            WHERE (YtHash = ${position.ytHash} AND AdGroupID > ${position.adGroupsId}
                OR YtHash > ${position.ytHash})
            ${if (maxYtHash != null) "AND YtHash <= $maxYtHash" else ""}
            ORDER BY YtHash, AdGroupID
            LIMIT $limit
            """
        val queryStarted = Instant.now().epochSecond
        val rows = context.ytProvider.getDynamicOperator(context.ytConfig.cluster)
            .selectRows(query)
            .yTreeRows
        val queryFinished = Instant.now().epochSecond
        logger.info("Query duration: ${queryFinished - queryStarted} sec")

        val updateTime = Instant.now().epochSecond
        val adGroupsWithYtHashList = rows
            .map {
                val ytHash = it.get("YtHash").get().longValue()
                AdGroupWithYtHash(buildAdGroupFromYtRow(it, updateTime), ytHash)
            }

        val finishTime = Instant.now().epochSecond
        logger.info("Duration: ${finishTime - startTimeStamp} sec")
        return adGroupsWithYtHashList
    }

    /**
     * Записывает сообщения в очередь
     */
    fun writeToQueue(adGroupsList: List<T>) {
        adGroupsList.chunked(WRITE_CHUNK_SIZE)
            .map { caesarQueueRequestFactory.getRequest(it, queuePath) }
            .forEach { doTransactionRequest(null, it) }
    }

    fun getYtHashMaxValue(): Long {
        val ytClient = context.ytProvider.getDynamicOperator(context.ytConfig.cluster).ytClient

        val schema = ytClient.getNode("$tablePath/@schema").join() // IGNORE-BAD-JOIN DIRECT-149116
        val expression = schema.listNode()
            .filter { it.mapNode().get("name").get().stringValue() == "YtHash" }
            .map { it.mapNode().get("expression").get().stringValue() }
            .first()
            .replace(" ", "")
        val matchedGroups = ytHashExpressionRegex.find(expression)
            ?: throw IllegalStateException("Unexpected YtHash expression: $expression")

        val (modulo) = matchedGroups.destructured
        return modulo.toLong() - 1
    }
}
