package ru.yandex.direct.bstransport.yt.repository.adgroup.resources

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Repository
import ru.yandex.adv.direct.adgroup.AdGroup
import ru.yandex.adv.direct.adgroup.OptionalInternalLevel
import ru.yandex.adv.direct.expression2.TargetingExpression
import ru.yandex.adv.direct.showcondition.RelevanceMatchData
import ru.yandex.adv.direct.showcondition.RfOptions
import ru.yandex.direct.bstransport.yt.repository.BsExportYtRepositoryContext
import ru.yandex.direct.bstransport.yt.repository.customColumn
import ru.yandex.direct.bstransport.yt.repository.protoColumn
import ru.yandex.direct.bstransport.yt.repository.stringColumn
import ru.yandex.direct.bstransport.yt.repository.uint64Column
import ru.yandex.direct.bstransport.yt.utils.ColumnMappingUtils.readProtoList
import ru.yandex.direct.bstransport.yt.utils.ColumnMappingUtils.readStringList
import ru.yandex.direct.bstransport.yt.utils.ColumnMappingUtils.writeProtoList
import ru.yandex.direct.bstransport.yt.utils.ColumnMappingUtils.writeStringList
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode

@Repository
open class AdGroupYtRepository(
    context: BsExportYtRepositoryContext
) : BaseAdGroupYtRepository(
    context
) {
    companion object {
        private val logger = LoggerFactory.getLogger(AdGroupYtRepository::class.java)
        const val WRITE_CHUNK_SIZE = 100_000
        private val ytHashExpressionRegex =
            "uint64\\(farm_hash\\(AdGroupID\\)%(\\d+)\\)".toRegex(RegexOption.IGNORE_CASE)
    }

    private val mappers = schemaWithMapping
    override fun getResourceColumnSchema(): List<AdGroupResourcesColumnMapping<out Any>> {
        return listOf(
            mapper(uint64Column("EngineID"),
                { it.engineId },
                optional { v, b -> b.engineId = v }),
            mapper(uint64Column("Type"), { it.type.toLong() }, optional { v, b -> b.type = v.toInt() }),
            mapper(stringColumn("Name"), { it.name }, optional { v, b -> b.name = v }),
            mapper(uint64Column("InternalLevel"),
                { if (it.hasInternalLevel() && it.internalLevel.hasValue()) it.internalLevel.value.toLong() else null },
                { value, builder ->
                    builder.internalLevel = OptionalInternalLevel.newBuilder()
                        .apply { if (value != null) setValue(value.toInt()) }
                        .build()
                }),
            mapper(customColumn("MinusPhrases"),
                { writeStringList(it.minusPhrasesList) },
                { value, builder ->
                    if (value != null) builder.addAllMinusPhrases(readStringList(value))
                }),
            mapper(customColumn("PageGroupTags"),
                { writeStringList(it.pageGroupTagsList) },
                { value, builder ->
                    if (value != null) builder.addAllPageGroupTags(readStringList(value))
                }),
            mapper(
                uint64Column("MatchPriority"),
                fromProtoToYtMapper = { if (it.hasMatchPriority()) it.matchPriority.toLong() else null },
                fromYtToProtoMapper = optional { v, b -> b.matchPriority = v.toInt() }
            ),
            mapper(customColumn("TargetTags"),
                { writeStringList(it.targetTagsList) },
                { value, builder ->
                    if (value != null) builder.addAllTargetTags(readStringList(value))
                }),
            mapper(
                stringColumn("ClickUrlTail"),
                fromProtoToYtMapper = { it.clickUrlTail },
                fromYtToProtoMapper = optional { v, b -> b.clickUrlTail = v },
            ),
            mapper(
                customColumn("Multipliers"),
                fromProtoToYtMapper = { writeProtoList(it.multipliersList) },
                fromYtToProtoMapper = optional { v, b -> b.addAllMultipliers(readProtoList(v)) }
            ),
            mapper(
                protoColumn("ShowConditions", TargetingExpression::class.java),
                { if (it.hasShowConditions()) it.showConditions else null },
                { value, builder -> if (value != null) builder.showConditions = value }
            ),
            mapper(uint64Column("ContextID"), { it.contextId }, optional { v, b -> b.contextId = v }),
            mapper(
                uint64Column("SerpPlacementType"),
                fromProtoToYtMapper = { it.serpPlacementType.toLong() },
                fromYtToProtoMapper = optional { v, b -> b.serpPlacementType = v.toInt() },
            ),
            mapper(
                protoColumn("RelevanceMatchData", RelevanceMatchData::class.java),
                { if (it.hasRelevanceMatchData()) it.relevanceMatchData else null },
                { value, builder -> if (value != null) builder.relevanceMatchData = value }
            ),
            mapper(
                protoColumn("RfOptions", RfOptions::class.java),
                { if (it.hasRfOptions()) it.rfOptions else null },
                { value, builder -> if (value != null) builder.rfOptions = value }
            ),
            mapper(
                stringColumn("FieldToUseAsName"),
                { it.fieldToUseAsName },
                optional { v, b -> b.fieldToUseAsName = v }),
            mapper(
                stringColumn("FieldToUseAsBody"),
                { it.fieldToUseAsBody },
                optional { v, b -> b.fieldToUseAsBody = v }),
            mapper(
                uint64Column("FeedID"),
                { it.feedId },
                optional { v, b -> b.feedId = v }),
        )
    }

    override fun buildAdGroupFromYtRow(node: YTreeMapNode, updateTime: Long): AdGroup {
        val builder = AdGroup.newBuilder()
        mappers.forEach { mapper -> mapper.mapToProtoBuilder(builder, node) }
        builder.updateTime = updateTime
        return builder.build()
    }
}
