package ru.yandex.direct.bstransport.yt.repository.feeds;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.springframework.stereotype.Component;

import ru.yandex.adv.direct.feeds.Feed;
import ru.yandex.adv.direct.feeds.FeedAccess;
import ru.yandex.direct.ytcomponents.config.BsExportYtDynConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

@Component
public class FeedsYtRepository {
    private final YtProvider ytProvider;
    private final YtCluster ytCluster;
    private final String feedsTablePath;
    private final String feedsAccessTablePath;

    private final TableSchema feedsTableSchema;
    private final TableSchema feedsAccessTableSchema;

    public FeedsYtRepository(YtProvider ytProvider, BsExportYtDynConfig ytConfig) {
        this.ytProvider = ytProvider;
        this.ytCluster = ytConfig.getCluster();
        this.feedsTablePath = ytConfig.getFeedsTable();
        this.feedsAccessTablePath = ytConfig.getFeedsAccessTable();
        this.feedsTableSchema = getFeedsTableSchema();
        this.feedsAccessTableSchema = getFeedsAccessTableSchema();

    }

    public void modifyFeeds(Collection<Feed> feeds) {
        ytProvider.getDynamicOperator(ytCluster).runInTransaction(tr -> modifyFeedsInternal(tr, feeds),
                new ApiServiceTransactionOptions(ETransactionType.TT_TABLET).setSticky(true));
    }

    public void delete(Collection<Long> ids) {
        ytProvider.getDynamicOperator(ytCluster).runInTransaction(tr -> deleteInternal(tr, ids, feedsTablePath,
                feedsTableSchema),
                new ApiServiceTransactionOptions(ETransactionType.TT_TABLET).setSticky(true));

        ytProvider.getDynamicOperator(ytCluster).runInTransaction(tr -> deleteInternal(tr, ids, feedsAccessTablePath,
                feedsAccessTableSchema),
                new ApiServiceTransactionOptions(ETransactionType.TT_TABLET).setSticky(true));
    }

    public void modifyFeedsAccess(Collection<FeedAccess> feedsAccess) {
        ytProvider.getDynamicOperator(ytCluster).runInTransaction(tr -> modifyFeedsAccessInternal(tr, feedsAccess),
                new ApiServiceTransactionOptions(ETransactionType.TT_TABLET).setSticky(true));
    }

    private void modifyFeedsInternal(ApiServiceTransaction transaction, Collection<Feed> feeds) {
        ModifyRowsRequest request = new ModifyRowsRequest(feedsTablePath, feedsTableSchema);
        request.setRequireSyncReplica(false);
        for (var feed : feeds) {
            request.addInsert(
                    List.of(
                            feed.getFeedId(),
                            feed.getClientId(),
                            feed.getBusinessType(),
                            feed.getFeedType(),
                            feed.getUrl(),
                            feed.getIsRemoteUtm()
                    )
            );
        }

        // todo timeout
        transaction.modifyRows(request).join(); // IGNORE-BAD-JOIN DIRECT-149116
    }

    private void deleteInternal(ApiServiceTransaction transaction, Collection<Long> ids, String tablePath,
                                TableSchema tableSchema) {
        ModifyRowsRequest request = new ModifyRowsRequest(tablePath, tableSchema.toKeys());
        request.setRequireSyncReplica(false);
        for (var id : ids) {
            request.addDelete(List.of(id));
        }

        // todo timeout
        transaction.modifyRows(request).join(); // IGNORE-BAD-JOIN DIRECT-149116
    }

    private void modifyFeedsAccessInternal(ApiServiceTransaction transaction, Collection<FeedAccess> feedsAccess) {
        ModifyRowsRequest request = new ModifyRowsRequest(feedsAccessTablePath, feedsAccessTableSchema);
        request.setRequireSyncReplica(false);
        for (var feedAccess : feedsAccess) {
            request.addInsert(
                    Arrays.asList(
                            feedAccess.getFeedId(),
                            feedAccess.getLogin(),
                            feedAccess.getLogin()
                    )
            );
        }

        // todo timeout
        transaction.modifyRows(request).join(); // IGNORE-BAD-JOIN DIRECT-149116
    }

    private TableSchema getFeedsTableSchema() {
        return new TableSchema.Builder()
                .addKey("FeedID", ColumnValueType.INT64)
                .addValue("ClientID", ColumnValueType.INT64)
                .addValue("BusinessType", ColumnValueType.STRING)
                .addValue("FeedType", ColumnValueType.STRING)
                .addValue("Url", ColumnValueType.STRING)
                .addValue("IsRemoteUtm", ColumnValueType.BOOLEAN)
                .build();
    }

    private TableSchema getFeedsAccessTableSchema() {
        return new TableSchema.Builder()
                .addKey("FeedID", ColumnValueType.INT64)
                .addValue("Login", ColumnValueType.STRING)
                .addValue("Password", ColumnValueType.STRING)
                .build();
    }
}
