package ru.yandex.direct.bstransport.yt.repository.resources

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.adv.direct.banner.resources.BannerResources
import ru.yandex.direct.bstransport.yt.repository.BsExportYtRepositoryContext
import ru.yandex.direct.bstransport.yt.repository.CaesarQueueRequestFactory
import ru.yandex.direct.bstransport.yt.repository.common.RowPosition
import ru.yandex.direct.utils.InterruptedRuntimeException
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode
import ru.yandex.misc.concurrent.TimeoutRuntimeException
import ru.yandex.misc.thread.ExecutionRuntimeException
import ru.yandex.yt.rpcproxy.ETransactionType
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest
import java.time.Instant
import java.util.concurrent.ExecutionException
import java.util.concurrent.TimeUnit
import java.util.concurrent.TimeoutException


data class ResourcesPosition(
    val ytHash: Long,
    val orderId: Long,
    val bannerId: Long
) : RowPosition()


data class BannerResourcesWithYtHash(
    val resources: BannerResources,
    val ytHash: Long
)

/**
 * Репозиторий для работы с таблицей и очередью yt для ресурсов баннера
 * Работает со всей строкой таблицы, не деля ее на подрепозитории
 */
@Component
open class BannerResourcesYtCommonRepository(private val context: BsExportYtRepositoryContext,
                                             private val caesarQueueRequestFactory: CaesarQueueRequestFactory,
                                             repositories: List<BaseBannerResourcesYtRepository>) {
    companion object {
        private val logger = LoggerFactory.getLogger(BannerResourcesYtCommonRepository::class.java)
        const val WRITE_CHUNK_SIZE = 100_000
        private val ytHashExpressionRegex = "uint64\\(farm_hash\\(OrderID\\)%(\\d+)\\)".toRegex(RegexOption.IGNORE_CASE)
    }

    private val mappers = getMappers(repositories)
    private fun getMappers(repositories: List<BaseBannerResourcesYtRepository>): List<(BannerResources.Builder, YTreeMapNode) -> Unit> {
        return repositories
            .flatMap { it.schemaWithMapping }
            .distinctBy { it.columnDescription.name }
            .map { { builder: BannerResources.Builder, row -> it.mapToProtoBuilder(builder, row) } }
    }

    /**
     * Читает из сортированной таблицы протобуф с ресурсами
     * Выбирает те, у которых OrderID > param.orderId или OrderID = param.orderId AND BannerID > param.bannerID
     * Фильтрует те, у которых DeltedTime not null
     * В поле UpdateTime записывает текущее время
     */
    fun getBannerResourcesForResync(position: ResourcesPosition, limit: Int, maxYtHash: Long? = null): List<BannerResourcesWithYtHash> {

        val startTimeStamp = Instant.now().epochSecond
        val query = """
            * FROM [${context.ytConfig.bannerResourcesTable}]
            WHERE (YtHash = ${position.ytHash} AND OrderID = ${position.orderId} AND BannerID > ${position.bannerId}
                OR YtHash = ${position.ytHash} AND OrderID > ${position.orderId}
                OR YtHash > ${position.ytHash})
            AND NOT is_null(ExportID) AND NOT is_null(IterID) AND NOT is_null(AdGroupID)
            ${if (maxYtHash != null) "AND YtHash <= $maxYtHash" else ""}
            AND IS_NULL(DeletedTime)
            ORDER BY YtHash, OrderID, BannerID
            LIMIT $limit
            """
        val queryStarted = Instant.now().epochSecond
        val rows = context.ytProvider.getDynamicOperator(context.ytConfig.cluster)
            .selectRows(query)
            .yTreeRows
        val queryFinished = Instant.now().epochSecond
        logger.info("Query duration: ${queryFinished - queryStarted} sec")

        val updateTime = Instant.now().epochSecond
        val bannerResourcesWithYtHashList = rows
            .map {
                val ytHash = it.get("YtHash").get().longValue()
                val builder = BannerResources.newBuilder()
                mappers.forEach { mapper -> mapper(builder, it) }
                builder.updateTime = updateTime
                BannerResourcesWithYtHash(builder.build(), ytHash)
            }

        val finishTime = Instant.now().epochSecond
        logger.info("Duration: ${finishTime - startTimeStamp} sec")
        return bannerResourcesWithYtHashList
    }

    /**
     * Записывает ресурсы в очередь
     */
    fun writeBannerResourcesToQueue(bannerResourcesList: List<BannerResources>) {
        bannerResourcesList.chunked(WRITE_CHUNK_SIZE)
            .map { caesarQueueRequestFactory.getRequest(it, context.ytConfig.bannerResourcesQueue) }
            .forEach { doTransactionRequest(it) }
    }

    open fun getYtHashMaxValue(): Long {
        val ytClient = context.ytProvider.getDynamicOperator(context.ytConfig.cluster).ytClient

        val schema = ytClient.getNode("${context.ytConfig.bannerResourcesTable}/@schema")
                .join() // IGNORE-BAD-JOIN DIRECT-149116
        val expression = schema.listNode()
            .filter { it.mapNode().get("name").get().stringValue() == "YtHash" }
            .map { it.mapNode().get("expression").get().stringValue() }
            .first()
            .replace(" ", "")
        val matchedGroups = ytHashExpressionRegex.find(expression)
            ?: throw IllegalStateException("Unexpected YtHash expression: $expression")

        val (modulo) = matchedGroups.destructured
        return modulo.toLong() - 1
    }

    private fun doTransactionRequest(request: ModifyRowsRequest) {
        context.ytProvider.getDynamicOperator(context.ytConfig.cluster).runInTransaction({ tr -> sendRequest(tr, request) },
            ApiServiceTransactionOptions(ETransactionType.TT_TABLET).setSticky(true))
    }

    private fun sendRequest(transaction: ApiServiceTransaction, request: ModifyRowsRequest) {
        try {
            transaction.modifyRows(request)[context.ytConfig.transactionTimeout.seconds, TimeUnit.SECONDS]
        } catch (e: InterruptedException) {
            Thread.currentThread().interrupt()
            throw InterruptedRuntimeException(e)
        } catch (e: ExecutionException) {
            throw ExecutionRuntimeException(e)
        } catch (e: TimeoutException) {
            throw TimeoutRuntimeException(e)
        }
    }
}
