package ru.yandex.direct.bstransport.yt.utils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import com.google.protobuf.CodedOutputStream;
import com.google.protobuf.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.io.IoUtils;

/**
 * Сериализация списка сообщений protobuf в формат protoseq
 * Сообщения разбиваются на N protoseq'ов по следующему принципу:
 * <li>В каждой последовательности не более {@code chunkSize} сообщений</li>
 * <li>Если в последовательности больше одного сообщения, её размер не больше {@code chunkSizeKilobytes} килобайтов</li>
 * <li>В последовательности есть как минимум одно сообщение</li>
 *
 * @see <a href="https://wiki.yandex-team.ru/logfeller/splitter/protoseq/">Формат Protoseq</a>
 */
public class ProtoSeqSerializer {
    private static final Logger logger = LoggerFactory.getLogger(ProtoSeqSerializer.class);
    private final int chunkSize;
    private final int chunkSizeBytes;

    public ProtoSeqSerializer(int chunkSize, int chunkSizeKilobytes) {
        this.chunkSize = chunkSize;
        this.chunkSizeBytes = chunkSizeKilobytes * 1024;
    }

    static final byte[] MAGIC = new byte[]{
            (byte) 0x1F, (byte) 0xF7, (byte) 0xF7, (byte) 0x7E, (byte) 0xBE, (byte) 0xA6, (byte) 0x5E, (byte) 0x9E,
            (byte) 0x37, (byte) 0xA6, (byte) 0xF6, (byte) 0x2E, (byte) 0xFE, (byte) 0xAE, (byte) 0x47, (byte) 0xA7,
            (byte) 0xB7, (byte) 0x6E, (byte) 0xBF, (byte) 0xAF, (byte) 0x16, (byte) 0x9E, (byte) 0x9F, (byte) 0x37,
            (byte) 0xF6, (byte) 0x57, (byte) 0xF7, (byte) 0x66, (byte) 0xA7, (byte) 0x06, (byte) 0xAF, (byte) 0xF7
    };

    public List<byte[]> serialize(List<? extends Message> messages) {
        List<byte[]> protoseqs = new ArrayList<>();
        int chunkBegin = 0;
        while (chunkBegin < messages.size()) {
            int seqTotalSize = 0;
            seqTotalSize += messages.get(chunkBegin).getSerializedSize() + 4 + MAGIC.length;
            int i = chunkBegin + 1;
            while (i < messages.size() && i - chunkBegin < chunkSize) {
                int size = messages.get(i).getSerializedSize();
                int newSeqTotalSize = seqTotalSize + size + 4 + MAGIC.length;
                if (newSeqTotalSize > chunkSizeBytes) {
                    break;
                }
                seqTotalSize = newSeqTotalSize;
                ++i;
            }
            byte[] protoseq = serialize(messages, chunkBegin, i, seqTotalSize);
            protoseqs.add(protoseq);
            chunkBegin = i;
        }
        return protoseqs;
    }

    public int getChunkSize() {
        return chunkSize;
    }

    public int getChunkSizeBytes() {
        return chunkSizeBytes;
    }

    /// Сериализация подмассива в protoseq. Размер protoseq уже подсчитан
    private byte[] serialize(List<? extends Message> messages, int from, int to, int serializedSize) {
        if (serializedSize > chunkSizeBytes) {
            logger.warn("Protoseq size ({}) exceeded limit ({})", serializedSize, chunkSizeBytes);
        }
        byte[] array = new byte[serializedSize];
        CodedOutputStream os = CodedOutputStream.newInstance(array);
        try {
            for (int i = from; i < to; i++) {
                int size = messages.get(i).getSerializedSize();
                os.writeFixed32NoTag(size);
                messages.get(i).writeTo(os);
                os.writeRawBytes(MAGIC);
            }
            os.checkNoSpaceLeft();
        } catch (IOException e) {
            throw IoUtils.translate(e);
        }
        return array;
    }
}
