package ru.yandex.direct.bsexport.iteration;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bsexport.iteration.container.ExportBatchLimits;
import ru.yandex.direct.bsexport.iteration.container.ExportCandidatesSelectionCriteria;
import ru.yandex.direct.bsexport.iteration.model.BsExportInstructions;
import ru.yandex.direct.bsexport.iteration.repository.CampaignsStatusBsSyncRepository;
import ru.yandex.direct.bsexport.iteration.repository.CandidatesRepository;
import ru.yandex.direct.core.entity.bs.export.model.CampaignIdWithSyncValue;
import ru.yandex.direct.core.entity.bs.export.model.WorkerPurpose;
import ru.yandex.direct.core.entity.bs.export.model.WorkerSpec;
import ru.yandex.direct.core.entity.bs.export.model.WorkerType;
import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportCandidateInfo;
import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportQueueInfo;
import ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportQueueRepository;
import ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportSpecialsRepository;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;


/**
 * Одна итерация экспорта данных в БК.
 * <p>
 * Содержит в себе параметры: <ul>
 * <li>шард</li>
 * <li>id воркера</li>
 * <li>ограничения на размер отправляемых данных</li>
 * </ul>
 * Также хранит данные (что требуется по ним отправить) об обрабатываемых текущим воркером кампаниям
 * и удобные методы по работе с ними (разблокировка в очереди, отметка данных синхронизированными)
 * <p>
 * После завершения итерации контекст нужно закрывать для освобождения залоченных кампаний
 */
@ParametersAreNonnullByDefault
public class BsExportIterationContext implements AutoCloseable {
    final BsExportIterationFactory factory;
    final ExportCandidatesSelectionCriteria selectionCriteria;
    private final Logger logger = LoggerFactory.getLogger(BsExportIterationContext.class);
    private final BsExportQueueRepository bsExportQueueRepository;
    private final BsExportSpecialsRepository bsExportSpecialsRepository;
    private final CandidatesRepository candidatesRepository;

    private final CampaignsStatusBsSyncRepository campaignsStatusBsSyncRepository;

    private final long workerId;
    private final WorkerType workerType;
    private final WorkerSpec workerSpec;
    private final LinkedHashMap<Long, BsExportInstructions> toDo;

    BsExportIterationContext(BsExportIterationFactory factory, ExportCandidatesSelectionCriteria selectionCriteria) {
        this.factory = factory;
        this.selectionCriteria = selectionCriteria;

        bsExportQueueRepository = factory.bsExportQueueRepository;
        bsExportSpecialsRepository = factory.bsExportSpecialsRepository;
        candidatesRepository = factory.candidatesRepository;
        workerSpec = this.selectionCriteria.getWorkerSpec();
        workerType = this.selectionCriteria.getWorkerType();
        workerId = this.selectionCriteria.getWorkerId();
        campaignsStatusBsSyncRepository = factory.campaignsStatusBsSyncRepository;
        toDo = new LinkedHashMap<>();
    }

    /**
     * Инициализация итерации. Здесь происходит:<ul>
     * <li>выбор из очереди подходящих кампаний на отправку и их дофильтрация</li>
     * <li>блокировка (на себя) отобранных кампаний в очереди</li>
     * <li>взятие и освобождение локов для миграции на новую схему зачислений</li>
     * <li>освобождение лишних кампаний</li>
     * <li>снятие с кампаний нерелевантной "назначенной очереди" (bs_export_specials)</li>
     * </ul>
     * В результате этого в контексте формируется словарь "что нужно делать с кампаниями",
     * состоящий из залоченных воркером кампаний и инструкций по ним.
     */
    void initialize() {
        List<BsExportCandidateInfo> candidatesForExport =
                candidatesRepository.getCandidatesForExport(selectionCriteria);
        processCandidates(candidatesForExport);

        Map<Long, BsExportQueueInfo> lockedCampaigns =
                candidatesRepository.getLockedCampaignsInfo(selectionCriteria);
        processLockedCampaigns(lockedCampaigns);

        // это могут быть кампании, оставшиеся от предыдущих запусков и не проходящие по лимитам
        unlockCampaignsWithoutInstructions();
        logNotOperatingCampaignIds();
    }

    private void processCandidates(List<BsExportCandidateInfo> candidatesForExport) {
        var campaignIdsToDeleteFromSpecials = new QueueTypeRemovingStep(this).filter(candidatesForExport);
        if (!campaignIdsToDeleteFromSpecials.isEmpty()) {
            bsExportSpecialsRepository.remove(getShard(), campaignIdsToDeleteFromSpecials);
        }

        CandidatesBatchingStep candidatesBatchingStep = new CandidatesBatchingStep(this);
        var batch = candidatesBatchingStep.batch(candidatesForExport);
        logger.info("{} candidates formed {}", candidatesForExport.size(), candidatesBatchingStep.getBatchStat());
        StreamEx.of(batch).mapToEntry(BsExportInstructions::getCampaignId).invert().forKeyValue(toDo::put);

        SumMigrationLockStep sumMigrationLockStep = new SumMigrationLockStep(this);
        var failedCampaignIds = sumMigrationLockStep.acquireLocks(toDo.values());
        failedCampaignIds.forEach(toDo::remove);

        bsExportQueueRepository.lockCampaigns(getShard(), toDo.keySet(), workerId);
        sumMigrationLockStep.releaseLocks();
    }

    private void processLockedCampaigns(Map<Long, BsExportQueueInfo> lockedCampaigns) {
        FindExcessCampaignsStep findExcessCampaignsStep = new FindExcessCampaignsStep(this, lockedCampaigns);

        // то, что не получилось залочить - сразу удаляем из контекста и больше не рассматриваем
        toDo.keySet().removeIf(findExcessCampaignsStep::campaignIsNotLocked);

        var excessCampaignIds = findExcessCampaignsStep.getExcessCampaignIds(toDo.keySet());
        if (!excessCampaignIds.isEmpty()) {
            unlockCampaigns(excessCampaignIds);
            excessCampaignIds.forEach(lockedCampaigns::remove);
        }

        var excessImageCampaignIds = findExcessCampaignsStep.getExcessImageCampaignIds(toDo.keySet());
        if (!excessImageCampaignIds.isEmpty()) {
            unlockCampaigns(excessCampaignIds);
            excessCampaignIds.forEach(lockedCampaigns::remove);
        }

        lockedCampaigns.values().forEach(this::clarifyInstructionsWithQueueInfo);
    }

    private void clarifyInstructionsWithQueueInfo(BsExportQueueInfo queueInfo) {
        Long campaignId = queueInfo.getCampaignId();

        BsExportInstructions instructions = toDo.computeIfAbsent(campaignId, this::createNoopInstruction);

        instructions.setSynchronizeValue(queueInfo.getSynchronizeValue());
        instructions.setNeedSendOther(calculateSendOtherFlag(instructions, queueInfo));
    }

    private BsExportInstructions createNoopInstruction(Long campaignId) {
        logger.trace("Got extra campaign from DB: {} - created no-op instruction", campaignId);
        return new BsExportInstructions()
                .withCampaignId(campaignId)
                .withNeedSendCamp(false)
                .withNeedSendData(false)
                .withNeedSendPrice(false)
                .withNeedFullExport(false);
    }

    boolean calculateSendOtherFlag(BsExportInstructions instructions, BsExportQueueInfo queueInfo) {
        return !instructions.getNeedSendCamp() && WorkerPurpose.ONLY_CAMPAIGNS.isApplicable(queueInfo)
                || !instructions.getNeedSendData() && WorkerPurpose.CONTEXTS_AND_BANNERS.isApplicable(queueInfo)
                || !instructions.getNeedSendPrice() && WorkerPurpose.ONLY_PRICES.isApplicable(queueInfo)
                || !instructions.getNeedFullExport() && WorkerPurpose.FULL_EXPORT.isApplicable(queueInfo);
    }

    /**
     * Разблокировать кампании, с которыми текущий воркер больше ничего не планирует делать
     */
    void unlockCampaignsWithoutInstructions() {
        Predicate<BsExportInstructions> hasNothingToOperate = instructions -> !hasSomethingToOperate(instructions);
        List<Long> campaignIds = toDo.values()
                .stream()
                .filter(hasNothingToOperate)
                .map(BsExportInstructions::getCampaignId)
                .collect(toList());

        if (campaignIds.isEmpty()) {
            return;
        }

        logger.info("Unlock campaigns without instructions {}", campaignIds);
        unlockCampaigns(campaignIds);
    }

    public void markCampaignsAsSynced(List<Long> campaignsIds) {
        if (campaignsIds.isEmpty()) {
            return;
        }
        campaignsStatusBsSyncRepository.setSynced(getShard(), campaignsIds);
        List<CampaignIdWithSyncValue> campaignsToResetNumFields = new ArrayList<>();
        List<CampaignIdWithSyncValue> campaignsIdsToRemoveWithSyncValues = new ArrayList<>();
        for (var campaignId : campaignsIds) {
            var campWithSyncValue = toDo.get(campaignId);
            // TODO DIRECT-120202: переделать частичную отметку кампании как отправленной
            if (!areAllObjectsSent(campaignId)) {
                campaignsToResetNumFields.add(campWithSyncValue);
            } else {
                campaignsIdsToRemoveWithSyncValues.add(campWithSyncValue);
            }
        }
        // TODO DIRECT-120237: вычитка кода: set_sync_campaigns
        bsExportQueueRepository.resetHandledFields(getShard(), campaignsToResetNumFields, workerId,
                workerType.getWorkerPurpose());
        deleteFromQueue(campaignsIdsToRemoveWithSyncValues);
        deleteFromSpecials(campaignsIdsToRemoveWithSyncValues, campaignsIds);
    }

    private void deleteFromQueue(List<CampaignIdWithSyncValue> campaignsIdsToRemoveWithSyncValues) {
        if (campaignsIdsToRemoveWithSyncValues.isEmpty()) {
            return;
        }
        var campaignsIds = mapList(campaignsIdsToRemoveWithSyncValues, CampaignIdWithSyncValue::getCampaignId);
        campaignsIds.forEach(toDo::remove);
        var deletedRowsCnt = bsExportQueueRepository.deleteIfSyncValueNotChanged(getShard(),
                campaignsIdsToRemoveWithSyncValues);
        if (deletedRowsCnt == campaignsIdsToRemoveWithSyncValues.size()) {
            return;
        }

        // TODO DIRECT-120197: двигать измененные в фоне кампании в конец очереди
        unlockCampaigns(campaignsIds);
    }

    private void deleteFromSpecials(List<CampaignIdWithSyncValue> campaignsIdsToRemoveWithSyncValues,
                                    List<Long> allCampaignsIds) {
        if (campaignsIdsToRemoveWithSyncValues.isEmpty() || !workerType.isTemporaryQueueType()) {
            return;
        }
        // TODO DIRECT-120238: разобраться с костылем про удаление par_type = camps_only в set_sync_campaigns
        var campaignsIdsToRemove = workerType == WorkerType.CAMPS_ONLY ? allCampaignsIds :
                mapList(campaignsIdsToRemoveWithSyncValues, CampaignIdWithSyncValue::getCampaignId);
        bsExportSpecialsRepository.remove(getShard(), campaignsIdsToRemove, workerType.getMappedQueueType());

    }

    private boolean areAllObjectsSent(Long campaignId) {
        if (workerType == WorkerType.CAMPS_ONLY) {
            return !toDo.get(campaignId).getNeedSendOther();
        } else {
            throw new IllegalStateException("Unsupported type to check if all objects sent " + workerType);
        }
    }

    /**
     * Будет ли текущий воркер делать что-то ещё с указанной кампанией
     *
     * @return {@code true} если остались запланированные операции отправки
     */
    boolean hasSomethingToOperate(BsExportInstructions instructions) {
        WorkerPurpose workerPurpose = selectionCriteria.getWorkerType().getWorkerPurpose();
        return workerPurpose.isDesignedToFullExport() && instructions.getNeedFullExport()
                || workerPurpose.isDesignedToSendCampaigns() && instructions.getNeedSendCamp()
                || workerPurpose.isDesignedToSendContextsAndBanners() && instructions.getNeedSendData()
                || workerPurpose.isDesignedToSendPrices() && instructions.getNeedSendPrice();
    }

    // аналог unlock с kind=all
    public void unlockCampaigns(Collection<Long> campaignIds) {
        bsExportQueueRepository.unlockCampaigns(getShard(), campaignIds, workerId);
        campaignIds.forEach(toDo::remove);
    }

    public void delayCampaigns(List<Long> campaignsIds) {
        if (campaignsIds.isEmpty()) {
            return;
        }
        var isFullExport = workerType == WorkerType.FULL_LB_EXPORT;
        bsExportQueueRepository.delayCampaigns(getShard(), campaignsIds, isFullExport);
    }

    public void moveToBuggy(List<Long> campaignsIds) {
        if (campaignsIds.isEmpty()) {
            return;
        }
        bsExportSpecialsRepository.moveToBuggy(getShard(), campaignsIds);
        delayCampaigns(campaignsIds);
    }

    private void unlockAllCampaigns() {
        bsExportQueueRepository.unlockCampaigns(getShard(), toDo.keySet(), workerId);
        toDo.clear();
    }

    private void logNotOperatingCampaignIds() {
        List<Long> notOperatingCampaignIds = selectionCriteria.getOnlyCampaignIds()
                .stream()
                .filter(campaignId -> !toDo.containsKey(campaignId))
                .collect(toList());
        if (notOperatingCampaignIds.isEmpty()) {
            return;
        }

        logger.warn("Campaigns {} have been requested but will not be processed. Possible " +
                        "reasons are: " +
                        "campaign is not in bs_export_queue table; " +
                        "campaign locked by different worker; " +
                        "campaign does not match to this worker (by bs_export_specials, type " +
                        "or queue stat)",
                notOperatingCampaignIds);
    }

    // for debug purpose only
    public Collection<BsExportInstructions> getInstructions() {
        return Collections.unmodifiableCollection(toDo.values());
    }

    /**
     * Получить id кампаний, для которых требуется отправка данных
     */
    public List<Long> getDataCampaignIds() {
        Predicate<BsExportInstructions> isApplicableToSendData = instructions
                -> instructions.getNeedSendCamp()
                || instructions.getNeedSendData()
                || instructions.getNeedFullExport();

        return toDo.values()
                .stream()
                .filter(isApplicableToSendData)
                .map(BsExportInstructions::getCampaignId)
                .collect(toList());
    }

    public int getShard() {
        return selectionCriteria.getShard();
    }

    public ExportBatchLimits getBatchLimits() {
        return selectionCriteria.getBatchLimits();
    }

    public long getWorkerId() {
        return workerId;
    }

    public WorkerType getWorkerType() {
        return workerType;
    }

    public WorkerSpec getWorkerSpec() {
        return workerSpec;
    }

    @Override
    public void close() {
        unlockAllCampaigns();
    }
}
