package ru.yandex.direct.bsexport.iteration;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import ru.yandex.direct.bsexport.iteration.model.BsExportInstructions;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.bs.export.model.WorkerPurpose;
import ru.yandex.direct.core.entity.bs.export.model.WorkerType;
import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportCandidateInfo;

/**
 * Работа с кандидатами на отправку - формирование пачки с учетом лимитов
 */
class CandidatesBatchingStep {
    private final WorkerType workerType;
    private final ExportBatchStat batchStat;

    CandidatesBatchingStep(BsExportIterationContext bsExportIterationContext) {
        workerType = bsExportIterationContext.selectionCriteria.getWorkerType();
        batchStat = new ExportBatchStat(bsExportIterationContext.getBatchLimits());
    }

    List<BsExportInstructions> batch(Collection<BsExportCandidateInfo> candidatesForExport) {
        return candidatesForExport.stream()
                .map(this::getCampaignInstructions)
                .filter(CandidatesBatchingStep::hasSomethingToOperate)
                .collect(Collectors.toList());
    }

    ExportBatchStat getBatchStat() {
        return batchStat;
    }

    private BsExportInstructions getCampaignInstructions(BsExportCandidateInfo candidateInfo) {
        boolean needFullExport = workerType.getWorkerPurpose().isDesignedToFullExport()
                && WorkerPurpose.FULL_EXPORT.isApplicable(candidateInfo)
                && batchStat.canFullExportAndIncrementStat(candidateInfo);

        boolean needSendData = workerType.getWorkerPurpose().isDesignedToSendContextsAndBanners()
                && WorkerPurpose.CONTEXTS_AND_BANNERS.isApplicable(candidateInfo)
                && batchStat.canSendDataAndIncrementStat(candidateInfo);

        // TODO DIRECT-110719: добавать в std отправку цен по кампаниям, которые уже попали в отправку
        boolean needSendPrice = workerType.getWorkerPurpose().isDesignedToSendPrices()
                && WorkerPurpose.ONLY_PRICES.isApplicable(candidateInfo)
                && (
                workerType.getWorkerPurpose().isDesignedToSendCampaigns()
                        || candidateInfo.getStatusBsSynced() == StatusBsSynced.YES)
                && batchStat.canSendPricesAndIncrementStat(candidateInfo);

        // отправку кампании проверяем после проверки цен и данных (так как их может быть много)
        // мы пока(?) не умеем отправлять только кампанию без данных, и лучше не взять кампанию совсем
        boolean needSendCamp = workerType.getWorkerPurpose().isDesignedToSendCampaigns()
                && WorkerPurpose.ONLY_CAMPAIGNS.isApplicable(candidateInfo)
                && batchStat.canSendCampaignAndIncrementStat(candidateInfo);

        return new BsExportInstructions()
                .withCampaignId(candidateInfo.getCampaignId())
                .withNeedFullExport(needFullExport)
                .withNeedSendCamp(needSendCamp)
                .withNeedSendData(needSendData)
                .withNeedSendPrice(needSendPrice);
    }

    private static boolean hasSomethingToOperate(BsExportInstructions instructions) {
        return instructions.getNeedSendCamp()
                || instructions.getNeedSendData()
                || instructions.getNeedSendPrice()
                || instructions.getNeedFullExport();
    }
}
