package ru.yandex.direct.bsexport.iteration;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.bsexport.iteration.container.ExportBatchLimits;
import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportQueueStat;

/**
 * Содержит логику "брать ли к отправке новый объект с учётом лимитов" и накопленную статистику.
 * <p>
 * Отправка данных - самая ресурсоёмкая операция, поэтому: отправка данных - выполняется первой (реализовано не здесь),
 * при исчерпании лимита по данным - другие объекты больше не набираются.
 * Это сделано для уменьшения простоя кампаний в очереди, когда кампания заблокирована но с ней ничего не делают.
 */
@ParametersAreNonnullByDefault
class ExportBatchStat {
    private final ExportBatchLimits limits;

    private int campaignsCount;
    private int contextsCount;
    private int bannersCount;
    private int keywordsCount;
    private int pricesCount;

    private boolean campaignsLimitReached;
    private boolean dataLimitReached;
    private boolean pricesLimitReached;

    public ExportBatchStat(ExportBatchLimits limits) {
        this.limits = limits;
    }

    private boolean internalCanSendCampaignAndIncrementStat(boolean sendCamp) {
        if (campaignsLimitReached || dataLimitReached) {
            return false;
        }

        int stat = sendCamp ? 1 : 0;
        if (this.campaignsCount + stat < limits.getCampaignsLimit()) {
            this.campaignsCount += stat;
            return true;
        }

        campaignsLimitReached = true;
        return false;
    }

    boolean canSendCampaignAndIncrementStat(BsExportQueueStat stat) {
        // на самом деле camps_num тоже "булевый" - равен 0 или 1.
        // исторически сложилось, что работаем с ним как с числом
        return internalCanSendCampaignAndIncrementStat(stat.getCampaignsCount() > 0);
    }

    boolean canFullExportAndIncrementStat(BsExportQueueStat stat) {
        return internalCanSendCampaignAndIncrementStat(stat.getNeedFullExport());
    }


    boolean canSendPricesAndIncrementStat(BsExportQueueStat stat) {
        if (pricesLimitReached || dataLimitReached) {
            return false;
        }

        boolean softLimit = pricesCount < limits.getPricesLimit() / 2;
        boolean hardLimit = pricesCount + stat.getPricesCount() < limits.getPricesLimit();
        if (softLimit || hardLimit) {
            pricesCount += stat.getPricesCount();
            return true;
        }

        pricesLimitReached = true;
        return false;
    }

    boolean canSendDataAndIncrementStat(BsExportQueueStat stat) {
        if (dataLimitReached) {
            return false;
        }

        boolean contextsSoftLimit = contextsCount < limits.getContextsLimit() / 2;
        boolean contextsHardLimit = contextsCount + stat.getContextsCount() < 3 * limits.getContextsLimit();

        boolean bannersSoftLimit = bannersCount < limits.getBannersLimit() / 2;
        boolean bannersHardLimit = bannersCount + stat.getBannersCount() < 3 * limits.getBannersLimit();

        boolean keywordsSoftLimit = keywordsCount < limits.getKeywordsLimit();
        boolean keywordsHardLimit = keywordsCount + stat.getKeywordsCount() < 3 * limits.getKeywordsLimit();

        if ((contextsSoftLimit || contextsHardLimit)
                && (bannersSoftLimit || bannersHardLimit)
                && (keywordsSoftLimit || keywordsHardLimit)) {
            contextsCount += stat.getContextsCount();
            bannersCount += stat.getBannersCount();
            keywordsCount += stat.getKeywordsCount();
            return true;
        }

        dataLimitReached = true;
        return false;
    }

    @Override
    public String toString() {
        return "ExportBatchStat{" +
                "campaignsCount=" + campaignsCount +
                ", contextsCount=" + contextsCount +
                ", bannersCount=" + bannersCount +
                ", keywordsCount=" + keywordsCount +
                ", pricesCount=" + pricesCount +
                '}';
    }
}
