package ru.yandex.direct.bsexport.iteration;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bsexport.iteration.container.ExportCandidatesSelectionCriteria;
import ru.yandex.direct.core.entity.bs.export.model.WorkerType;
import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportQueueInfo;

/**
 * Работа по выявлению "лишних" кампаний, которые нужно разблокировать
 */
@ParametersAreNonnullByDefault
class FindExcessCampaignsStep {
    private static final Logger logger = LoggerFactory.getLogger(FindExcessCampaignsStep.class);

    private final ExportCandidatesSelectionCriteria selectionCriteria;
    private final Map<Long, BsExportQueueInfo> lockedCampaigns;

    FindExcessCampaignsStep(BsExportIterationContext iterationContext, Map<Long, BsExportQueueInfo> lockedCampaigns) {
        this.lockedCampaigns = lockedCampaigns;
        this.selectionCriteria = iterationContext.selectionCriteria;
    }

    boolean campaignIsNotLocked(Long campaignId) {
        return !lockedCampaigns.containsKey(campaignId);
    }

    /**
     * Если получилось залочить больше, чем просили - то нужно разлочить лишнее.
     * При этом мы не хотим работать с кампаниями, которые мы не пытались залочить на текущей итерации,
     * потому что не проверяли их лимиты относительно "пачки.
     * Поэтому разблокируем все "неожииданные" кампании (чтобы не формировать пачку заново), и в этом
     * полагаемся на заполненность preferredCampaignIds в нужном порядке.∆
     *
     * @param preferredCampaignIds кампании которые мы хотим обрабатывать
     * @return id лишних кампаний, которые нужно разблокировать
     */
    Set<Long> getExcessCampaignIds(Collection<Long> preferredCampaignIds) {
        if (lockedCampaigns.size() <= selectionCriteria.getLimit()) {
            return Collections.emptySet();
        }

        var result = new HashSet<>(lockedCampaigns.keySet());
        preferredCampaignIds.stream()
                .filter(lockedCampaigns::containsKey)
                .limit(selectionCriteria.getLimit())
                .forEach(result::remove);

        logger.warn("locked {} campaigns, but limit was {}: {} excess",
                lockedCampaigns.size(), selectionCriteria.getLimit(), result.size());
        return result;
    }

    /**
     * В ре-экспорте самое объемное к отправке - MdsMeta в картиночных баннерах,
     * поэтому смотрим на то, сколько у нас картинок
     *
     * @param campaignIds кампании, которые мы хотим обрабатывать
     * @return id лишних кампаний, которые нужно разблокировать
     */
    List<Long> getExcessImageCampaignIds(Collection<Long> campaignIds) {
        if (selectionCriteria.getWorkerType() != WorkerType.FULL_LB_EXPORT) {
            return Collections.emptyList();
        }

        // TODO DIRECT-111645: учитывать количество картинок в fullLbExport
        List<Long> result = Collections.emptyList();

        if (!result.isEmpty()) {
            logger.warn("campaigns with a lot of images: {} excess", result.size());
        }

        return result;
    }
}
