package ru.yandex.direct.bsexport.iteration;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.bsexport.iteration.model.BsExportInstructions;
import ru.yandex.direct.core.entity.campaign.model.AggregatingSumStatus;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithTypeAndWalletId;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.wallet.model.WalletParamsModel;
import ru.yandex.direct.core.entity.walletparams.repository.WalletParamsRepository;
import ru.yandex.direct.redislock.DistributedLock;
import ru.yandex.direct.utils.CommonUtils;

import static com.google.common.base.Preconditions.checkState;

@ParametersAreNonnullByDefault
class SumMigrationLockStep {
    private final CampaignRepository campaignRepository;
    private final WalletParamsRepository walletParamsRepository;
    private final int shard;

    // ключи - номера кошельков на которые нужно брать блокировки, значения - номера кампаний под ним (и он сам)
    private final Map<Long, List<Long>> walletIdToCampaignIds;
    // взятые локи
    private final Set<DistributedLock> acquiredLocks;

    private boolean locked;

    SumMigrationLockStep(BsExportIterationContext iterationContext) {
        this.campaignRepository = iterationContext.factory.campaignRepository;
        this.walletParamsRepository = iterationContext.factory.walletParamsRepository;
        this.shard = iterationContext.getShard();

        walletIdToCampaignIds = new HashMap<>();
        acquiredLocks = new HashSet<>();
    }

    /**
     * Отобрать кампании для которых нужно брать лок
     */
    private void prepare(Collection<BsExportInstructions> candidates) {
        walletIdToCampaignIds.clear();
        acquiredLocks.clear();

        List<Long> campaignIds = candidates.stream()
                .filter(this::isApplicable)
                .map(BsExportInstructions::getCampaignId)
                .collect(Collectors.toList());

        if (campaignIds.isEmpty()) {
            return;
        }

        var campaignsWithTypeAndWalletId = campaignRepository.getCampaignsWithTypeAndWalletId(shard, campaignIds);

        Map<Long, List<Long>> candidateWallets = StreamEx.of(campaignsWithTypeAndWalletId)
                .mapToEntry(this::getWalletId, CampaignWithTypeAndWalletId::getId)
                .filterKeys(CommonUtils::isValidId)
                .grouping();

        var walletParams = walletParamsRepository.get(shard, candidateWallets.keySet());
        StreamEx.of(walletParams)
                .filter(this::canMigrate)
                .map(WalletParamsModel::getId)
                .mapToEntry(candidateWallets::get)
                .forKeyValue(walletIdToCampaignIds::put);
    }

    /**
     * Пробует взять Redis-лок на общие счета кампаний, чтобы можно было приостановить миграцию ОС отправляемых кампаний
     * путем выставления им {@code bs_export_queue.par_id}
     * <p>
     * Если лок взять не удалось, такие кампании отправлять нельзя.
     *
     * @return список идентификаторов кампаний, для которых не получилось взять лок
     */
    List<Long> acquireLocks(Collection<BsExportInstructions> candidates) {
        checkState(!locked);

        prepare(candidates);
        locked = true;

        if (walletIdToCampaignIds.isEmpty() || !isSumMigrationPossible()) {
            return Collections.emptyList();
        }

        // TODO DIRECT-110307: блокировки в redis для миграции на новую схему зачислений
        // написать настоящую реализация, с использованием AggregateMigrationRedisLockService (?) или массового метода
        // для всех ключей walletIdToCampaignIds взять локи (AggregateMigrationRedisLockService#lock)
        // кому получилось - добавить в acquiredLocks
        // кому нет - добавить значения (соответствующие этим ключам) в возвращаемый результат

        return Collections.emptyList();
    }

    void releaseLocks() {
        checkState(locked);

        // TODO DIRECT-110307: блокировки в redis для миграции на новую схему зачислений
        // что-то вроде такого
        // acquiredLocks.forEach(AggregateMigrationRedisLockService::unlock);

        locked = false;
    }

    boolean isSumMigrationPossible() {
        // TODO DIRECT-110307: блокировки в redis для миграции на новую схему зачислений
        // пошарить код с NotifyOrderService#readPpcPropertyWalletMigrationStateFlag
        // реализовать здесь Bs::ExportWorker::_is_sum_migration_possible
        return false;
    }

    /**
     * может ли понадобиться блокировка для отправляемой кампании
     */
    private boolean isApplicable(BsExportInstructions instructions) {
        return instructions.getNeedSendCamp() || instructions.getNeedSendData();
    }

    /**
     * получить id кошелька
     */
    private Long getWalletId(CampaignWithTypeAndWalletId campaign) {
        if (campaign.getType() == CampaignType.WALLET) {
            return campaign.getId();
        } else {
            return campaign.getWalletId();
        }
    }

    private boolean canMigrate(WalletParamsModel walletParams) {
        return walletParams.getAggregateMigrateStatus() == AggregatingSumStatus.NO;
    }
}
