package ru.yandex.direct.bsexport.iteration.container;

import java.util.Collections;
import java.util.Set;
import java.util.StringJoiner;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.bs.export.model.WorkerSpec;
import ru.yandex.direct.core.entity.bs.export.model.WorkerType;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Настройки для выбора кампаний из очереди экспорта
 */
@ParametersAreNonnullByDefault
public class ExportCandidatesSelectionCriteria {

    private final WorkerSpec workerSpec;
    private final int shard;
    private final Set<Long> onlyCampaignIds;
    private final ExportBatchLimits exportBatchLimits;
    private final int limit;
    private final int limitOverlap;
    private final boolean lockNewCampaigns;
    private final boolean skipLockedWallets;

    private ExportCandidatesSelectionCriteria(Builder builder) {
        this.workerSpec = builder.workerSpec;
        this.shard = builder.shard;
        this.onlyCampaignIds = Collections.unmodifiableSet(builder.onlyCampaignIds);
        this.exportBatchLimits = builder.exportBatchLimits;
        this.limit = builder.limit;
        this.limitOverlap = builder.limitOverlap;
        this.lockNewCampaigns = builder.lockNewCampaigns;
        this.skipLockedWallets = builder.skipLockedWallets;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private WorkerSpec workerSpec;
        private int shard;
        private Set<Long> onlyCampaignIds;
        private ExportBatchLimits exportBatchLimits;
        private int limit;
        private int limitOverlap;
        private boolean lockNewCampaigns;
        private boolean skipLockedWallets;

        private Builder() {
            // умолчания
            exportBatchLimits = ExportBatchLimits.DEFAULT;
            skipLockedWallets = true;
            lockNewCampaigns = true;
            onlyCampaignIds = Collections.emptySet();
        }

        public Builder setWorkerSpec(WorkerSpec workerSpec) {
            this.workerSpec = workerSpec;
            return this;
        }

        public Builder setShard(int shard) {
            this.shard = shard;
            return this;
        }

        /**
         * Ограничить выборку только указанными кампаниями
         */
        public Builder setOnlyCampaignIds(@Nonnull Iterable<Long> campaignIds) {
            this.onlyCampaignIds = StreamEx.of(campaignIds.spliterator()).toSet();
            return this;
        }

        /**
         * Задать параметры "пачки"
         */
        public Builder setExportBatchLimits(ExportBatchLimits exportBatchLimits) {
            this.exportBatchLimits = exportBatchLimits;
            return this;
        }

        /**
         * Задать лимит выборки
         */
        public Builder setLimit(int limit) {
            this.limit = limit;
            return this;
        }

        /**
         * Задать перекрытие (превышение) лимита при выборке из базы
         * Полезно использовать при маленьких значениях лимита
         */
        public Builder setLimitOverlap(int limitOverlap) {
            this.limitOverlap = limitOverlap;
            return this;
        }

        /**
         * Брать новые (еще никем не заблокированные) заказы.
         * Если нет, будут отобраны только заказы, залоченные на текущий воркер.
         */
        public Builder setLockNewCampaigns(boolean lockNewCampaigns) {
            this.lockNewCampaigns = lockNewCampaigns;
            return this;
        }

        /**
         * Задать настройку "не брать кампании из групповых заказов, часть из которых уже заблокирована".
         * Учитываются как сами кошельки, так и кампании под ними.
         * <p>
         * Полезно, так как при обновлении крутилка на своей стороне берет локи на каждый заказ группы.
         */
        public Builder setSkipLockedWallets(boolean skipLockedWallets) {
            this.skipLockedWallets = skipLockedWallets;
            return this;
        }

        public ExportCandidatesSelectionCriteria build() {
            checkArgument(shard > 0, "shard should be greater than 0");
            checkNotNull(workerSpec, "workerSpec should be defined");
            checkArgument(limit >= 0, "limit should not be negative");
            checkArgument(limitOverlap >= 0, "limit overlap should not be negative");
            checkNotNull(exportBatchLimits, "exportBatchLimits should be defined");

            if (limit == 0) {
                limit = exportBatchLimits.getCampaignsLimit();
            }

            return new ExportCandidatesSelectionCriteria(this);
        }
    }

    public int getShard() {
        return shard;
    }

    /**
     * Получить лимит.
     *
     * @return заданное явно значение или 2х {@link ExportBatchLimits#getCampaignsLimit()}
     */
    public int getLimit() {
        return limit;
    }

    /**
     * Получить ограничение на выборку из базы
     *
     * @return {@code limit + limitOverlap}
     */
    public int getLockLimit() {
        return limit + limitOverlap;
    }

    public Set<Long> getOnlyCampaignIds() {
        return onlyCampaignIds;
    }

    public boolean isLockNewCampaigns() {
        return lockNewCampaigns;
    }

    public boolean isSkipLockedWallets() {
        return skipLockedWallets;
    }

    public long getWorkerId() {
        return workerSpec.getWorkerId();
    }

    public WorkerType getWorkerType() {
        return workerSpec.getWorkerType();
    }

    public WorkerSpec getWorkerSpec() {
        return workerSpec;
    }

    public ExportBatchLimits getBatchLimits() {
        return exportBatchLimits;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", ExportCandidatesSelectionCriteria.class.getSimpleName() + "[", "]")
                .add("onlyCampaignIds=" + !onlyCampaignIds.isEmpty())
                .add("limit=" + getLimit())
                .add("lockNewCampaigns=" + lockNewCampaigns)
                .add("skipLockedWallets=" + skipLockedWallets)
                .add("workerSpec=" + workerSpec)
                .toString();
    }
}
