package ru.yandex.direct.bsexport.iteration.repository;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.bsexport.iteration.container.ExportCandidatesSelectionCriteria;
import ru.yandex.direct.core.entity.bs.export.model.WorkerPurpose;
import ru.yandex.direct.core.entity.bs.export.model.WorkerSpec;
import ru.yandex.direct.core.entity.bs.export.model.WorkerType;
import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportCandidateInfo;
import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportQueueInfo;
import ru.yandex.direct.core.entity.bs.export.queue.model.QueueType;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.dbschema.ppc.enums.BsExportSpecialsParType;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusactive;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperUtils;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportQueueRepository.EMPTY_STAT;
import static ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportQueueRepository.EXCLUDE_SPECIAL_PAR_TYPES;
import static ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportQueueRepository.createBaseMapperBuilder;
import static ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportQueueRepository.createQueueInfoMapper;
import static ru.yandex.direct.core.entity.bs.export.queue.service.BsExportQueueService.WORKER_TYPES_NOT_RESTRICTED_BY_LOCKED_WALLETS;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignRepository.WALLETS;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignRepository.campaignBalance;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignRepository.getNoWalletOrWalletHasOrderIdCondition;
import static ru.yandex.direct.dbschema.ppc.Tables.BS_EXPORT_QUEUE;
import static ru.yandex.direct.dbschema.ppc.Tables.BS_EXPORT_SPECIALS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbutil.SqlUtils.ID_NOT_SET;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;

/**
 * Репозиторий для работы получения кампаний-кандидатов из очереди.
 * Идеологическое продолжение {@link ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportQueueRepository}
 */
@Repository
@ParametersAreNonnullByDefault
public class CandidatesRepository {
    private static final Logger logger = LoggerFactory.getLogger(CandidatesRepository.class);

    private static final List<CampaignsType> INTERNAL_ADS_CAMPAIGN_TYPES = StreamEx.of(CampaignTypeKinds.INTERNAL)
            .map(CampaignType::toSource)
            .toImmutableList();

    private static final List<Long> WORKER_IDS_NOT_RESTRICTED_BY_LOCKED_WALLETS = StreamEx.of(WorkerSpec.values())
            .mapToEntry(WorkerSpec::getWorkerType, WorkerSpec::getWorkerId)
            .filterKeys(WORKER_TYPES_NOT_RESTRICTED_BY_LOCKED_WALLETS::contains)
            .values()
            .toList();

    private static final Field<Long> WALLET_CID =
            JooqMapperUtils.mysqlIf(CAMPAIGNS.TYPE.eq(CampaignsType.wallet), CAMPAIGNS.CID, CAMPAIGNS.WALLET_CID);

    private static final Condition HAS_CAMPS_STAT = BS_EXPORT_QUEUE.CAMPS_NUM.gt(EMPTY_STAT);

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<BsExportQueueInfo> queueInfoJooqMapper;
    private final JooqMapperWithSupplier<BsExportCandidateInfo> candidateInfoJooqMapper;

    @Autowired
    public CandidatesRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.queueInfoJooqMapper = createQueueInfoMapper();
        this.candidateInfoJooqMapper = createCandidateInfoMapper();
    }

    /**
     * Получить сведения о кампаниях, залоченных в очереди экспорта в БК.
     * Выбирает все кампании, заблокированные указанным воркеров (или только те из них, что указаны в критерии)
     *
     * @param selectionCriteria настройки выборки (шард, воркер, лимиты, режимы)
     * @return словарь, в котором ключами являются номера кампаний, а значениями - информация о них
     */
    public Map<Long, BsExportQueueInfo> getLockedCampaignsInfo(ExportCandidatesSelectionCriteria selectionCriteria) {
        Condition campaignIdsCondition = getCampaignIdsOnlyCondition(selectionCriteria);

        var result = dslContextProvider.ppc(selectionCriteria.getShard())
                .select(queueInfoJooqMapper.getFieldsToRead())
                .from(BS_EXPORT_QUEUE)
                .where(BS_EXPORT_QUEUE.PAR_ID.eq(selectionCriteria.getWorkerId())
                        .and(campaignIdsCondition))
                .fetchMap(BS_EXPORT_QUEUE.CID, queueInfoJooqMapper::fromDb);

        logger.trace("locked campaignIds: {}", result.keySet());
        return result;
    }

    private Condition makeCandidatesForExportCondition(ExportCandidatesSelectionCriteria selectionCriteria) {
        Condition alreadyLockedCampaigns = BS_EXPORT_QUEUE.PAR_ID.eq(selectionCriteria.getWorkerId());
        Condition campaignIdsCondition = getCampaignIdsOnlyCondition(selectionCriteria);
        Condition newCampaignsToLock = getConditionForUnlockedCampaigns(selectionCriteria);

        return alreadyLockedCampaigns.or(newCampaignsToLock)
                .and(campaignIdsCondition);
    }

    /**
     * Получить часть условия для {@link #makeCandidatesForExportCondition},
     * отвечающую за работу только с конкретными номерами кампаний.
     * Используется в тестах и для отладки.
     */
    private static Condition getCampaignIdsOnlyCondition(ExportCandidatesSelectionCriteria selectionCriteria) {
        Set<Long> onlyCampaignIds = selectionCriteria.getOnlyCampaignIds();
        if (onlyCampaignIds.isEmpty()) {
            return DSL.trueCondition();
        } else {
            return BS_EXPORT_QUEUE.CID.in(selectionCriteria.getOnlyCampaignIds());
        }
    }

    /**
     * Получить часть условия для {@link #makeCandidatesForExportCondition},
     * отвечающую за отбор ещё никем не отправляемых кампаний
     */
    private Condition getConditionForUnlockedCampaigns(ExportCandidatesSelectionCriteria selectionCriteria) {
        if (!selectionCriteria.isLockNewCampaigns()) {
            return DSL.falseCondition();
        }

        Condition withoutLockedWallets = getLockedWalletsCondition(selectionCriteria);
        Condition workerTypeCondition = getExportSpecialsCondition(selectionCriteria);
        Condition internalAdsOrNot = getConditionToSeparateInternalAds(selectionCriteria);
        Condition workerSpecializationCondition = getWorkerSpecializationCondition(selectionCriteria);
        Condition campaignHasNoWalletOrWalletHasOrderID = getNoWalletOrWalletHasOrderIdCondition(CAMPAIGNS, WALLETS);

        return BS_EXPORT_QUEUE.PAR_ID.isNull()
                .and(withoutLockedWallets)
                .and(workerTypeCondition)
                .and(internalAdsOrNot)
                .and(workerSpecializationCondition)
                .and(campaignHasNoWalletOrWalletHasOrderID)
                ;
    }

    /**
     * Получить часть условия для {@link #getConditionForUnlockedCampaigns} отвечающую за то, чтобы групповые заказы
     * отправлялись одновременно только одним (или нет) воркером.
     * <p>
     * Зачем это нужно: при отправке данных по заказу в bssoap, в случае общего счета, крутилка берет позаказные локи
     * на все (а не только присланные) заказы группы. Попытки (с нашей сторону) отправлять заказы одной группы
     * разными воркерами - впустую расходуют время, из-за получения ошибки блокировки
     * <p>
     * Оригинальная реализация:
     * https://st.yandex-team.ru/DIRECT-35156
     * https://st.yandex-team.ru/DIRECT-42238
     */
    private Condition getLockedWalletsCondition(ExportCandidatesSelectionCriteria selectionCriteria) {
        if (!selectionCriteria.isSkipLockedWallets()) {
            return DSL.trueCondition();
        }

        Condition withoutWallet = CAMPAIGNS.WALLET_CID.eq(ID_NOT_SET)
                .and(CAMPAIGNS.TYPE.ne(CampaignsType.wallet));

        List<Long> lockedWalletIds = getLockedWallets(selectionCriteria.getShard(), selectionCriteria.getWorkerId(),
                selectionCriteria.getLockLimit() * 10);

        return withoutWallet.or(WALLET_CID.notIn(lockedWalletIds));
    }

    /**
     * Получить часть условия для {@link #getConditionForUnlockedCampaigns} отвечающую за связь между типом воркера
     * и назначенным кампании типом очереди (bs_export_specials)
     */
    private static Condition getExportSpecialsCondition(ExportCandidatesSelectionCriteria selectionCriteria) {
        switch (selectionCriteria.getWorkerType()) {
            // Берут только свои заказы
            case FAST:
            case HEAVY:
            case DEV1:
            case DEVPRICE1:
            case DEV2:
            case DEVPRICE2:
            case BUGGY:
            case PREPROD:
            case CAMPS_ONLY:
                QueueType queueType = checkNotNull(selectionCriteria.getWorkerType().getMappedQueueType(),
                        "WorkerType %s has no mapping into queue type (bs_export_specials.par_type)");
                return BS_EXPORT_SPECIALS.PAR_TYPE.eq(QueueType.toSource(queueType));

            case CAMP:
                return EXCLUDE_SPECIAL_PAR_TYPES.and(BS_EXPORT_SPECIALS.PAR_TYPE.isNull()
                        .or(BS_EXPORT_SPECIALS.PAR_TYPE
                                .notIn(BsExportSpecialsParType.preprod, BsExportSpecialsParType.camps_only)));


            // Берут только не-специальные заказы
            case STD:
            case STDPRICE:
            case INTERNAL_ADS:
                return BS_EXPORT_SPECIALS.PAR_TYPE.isNull();

            // берет любые заказы (кроме раработческих и совсем специальных)
            case FULL_LB_EXPORT:
                return EXCLUDE_SPECIAL_PAR_TYPES;

            default:
                throw new UnsupportedOperationException("WorkerType " + selectionCriteria.getWorkerType()
                        + " has no specified condition for queue type");
        }
    }

    /**
     * Получить часть условия для {@link #getConditionForUnlockedCampaigns} отвечающую за то,
     * кампании с какими характеристиками будут выбраны
     */
    private static Condition getWorkerSpecializationCondition(ExportCandidatesSelectionCriteria selectionCriteria) {
        Condition workerPurposeCondition = getWorkerPurposeCondition(selectionCriteria);

        switch (selectionCriteria.getWorkerType()) {
            // Исключая несинхронные кампании, так как состоения автобюджета нужно передавать раньше цен
            case STDPRICE:
            case DEVPRICE1:
            case DEVPRICE2:
                return workerPurposeCondition.and(CAMPAIGNS.STATUS_BS_SYNCED.eq(CampaignsStatusbssynced.Yes));

            // Отправляем новые общие счета, остановку кампаний или окончание денег
            // Последний раз логику меняли в https://st.yandex-team.ru/DIRECT-37862
            case CAMP:
                Condition campaignIsWalletWithoutOrderID = CAMPAIGNS.TYPE.eq(CampaignsType.wallet)
                        .and(CAMPAIGNS.ORDER_ID.eq(ID_NOT_SET));
                Condition campaignIsActive = CAMPAIGNS.STATUS_ACTIVE.eq(CampaignsStatusactive.Yes);
                Condition campaignShouldStop = CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.No)
                        .or(campaignBalance(CAMPAIGNS, WALLETS).le(BigDecimal.ZERO));

                return workerPurposeCondition.and(campaignIsWalletWithoutOrderID.or(campaignIsActive.and(campaignShouldStop)));

            default:
                return workerPurposeCondition;
        }
    }

    /**
     * Получать часть условия для {@link #getWorkerSpecializationCondition} отвечающую
     * за условие на статистику в очереди
     */
    private static Condition getWorkerPurposeCondition(ExportCandidatesSelectionCriteria selectionCriteria) {
        WorkerPurpose workerPurpose = selectionCriteria.getWorkerType().getWorkerPurpose();

        List<Condition> conditions = new ArrayList<>(4);
        if (workerPurpose.isDesignedToFullExport()) {
            conditions.add(BS_EXPORT_QUEUE.IS_FULL_EXPORT.eq(1L));
        }
        if (workerPurpose.isDesignedToSendCampaigns()) {
            conditions.add(HAS_CAMPS_STAT);
        }
        if (workerPurpose.isDesignedToSendPrices()) {
            conditions.add(BS_EXPORT_QUEUE.PRICES_NUM.gt(EMPTY_STAT));
        }
        if (workerPurpose.isDesignedToSendContextsAndBanners()) {
            conditions.add(BS_EXPORT_QUEUE.BANNERS_NUM.gt(EMPTY_STAT)
                    .or(BS_EXPORT_QUEUE.CONTEXTS_NUM.gt(EMPTY_STAT))
                    .or(BS_EXPORT_QUEUE.BIDS_NUM.gt(EMPTY_STAT)));
        }

        Optional<Condition> result = conditions.stream().reduce(Condition::or);
        if (result.isEmpty()) {
            throw new UnsupportedOperationException("WorkerType " + selectionCriteria.getWorkerType()
                    + " has no specified conditions for campaigns in queue");
        }

        return result.get();
    }

    /**
     * Получить часть условия для {@link #getConditionForUnlockedCampaigns},
     * отвечающую за разделение на "внутренняя реклама" и всё остальное.
     * <p>
     * Это требуется для воркеров, работающих с bssoap, так как там один EngineID на весь запрос,
     * а у этих типов он - разный.
     */
    private static Condition getConditionToSeparateInternalAds(ExportCandidatesSelectionCriteria selectionCriteria) {
        switch (selectionCriteria.getWorkerType()) {
            case FULL_LB_EXPORT:
                return DSL.trueCondition();

            case INTERNAL_ADS:
                return CAMPAIGNS.TYPE.in(INTERNAL_ADS_CAMPAIGN_TYPES);

            default:
                return CAMPAIGNS.TYPE.notIn(INTERNAL_ADS_CAMPAIGN_TYPES);
        }
    }

    /**
     * Получить кампании, подходящие для блокировки и последюущей отправки указанным воркером
     *
     * @param selectionCriteria настройки выборки (шард, воркер, лимиты, режимы)
     */
    public List<BsExportCandidateInfo> getCandidatesForExport(ExportCandidatesSelectionCriteria selectionCriteria) {
        return getCandidatesForExport(selectionCriteria, null);
    }

    /**
     * Только для тестов: разрешаем переопределить dslContext
     */
    List<BsExportCandidateInfo> getCandidatesForExport(ExportCandidatesSelectionCriteria selectionCriteria,
                                                       @Nullable DSLContext overrideDslContext) {
        logger.debug("get candidate campaigns for export with {}", selectionCriteria);

        Condition candidatesCondition = makeCandidatesForExportCondition(selectionCriteria);

        // В первую очередь получаем уже заблокированные нами заказы
        Field<Integer> preferAlreadyLocked = DSL.case_()
                .value(BS_EXPORT_QUEUE.PAR_ID)
                .when(selectionCriteria.getWorkerId(), 0)
                .else_(1);
        // ре-экспорт имеет собственную сквозную сортировку по времени
        Field<LocalDateTime> sequenceTimeField = selectionCriteria.getWorkerType() == WorkerType.FULL_LB_EXPORT
                ? BS_EXPORT_QUEUE.FULL_EXPORT_SEQ_TIME
                : BS_EXPORT_QUEUE.SEQ_TIME;

        DSLContext dslContext = overrideDslContext;
        if (dslContext == null) {
            dslContext = dslContextProvider.ppc(selectionCriteria.getShard());
        }

        return dslContext
                .select(candidateInfoJooqMapper.getFieldsToRead())
                .from(BS_EXPORT_QUEUE)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BS_EXPORT_QUEUE.CID))
                .leftJoin(WALLETS).on(WALLETS.CID.eq(CAMPAIGNS.WALLET_CID))
                .leftJoin(BS_EXPORT_SPECIALS).on(BS_EXPORT_SPECIALS.CID.eq(BS_EXPORT_QUEUE.CID))
                .where(candidatesCondition)
                .orderBy(preferAlreadyLocked, sequenceTimeField, BS_EXPORT_QUEUE.QUEUE_TIME, BS_EXPORT_QUEUE.CID)
                .limit(selectionCriteria.getLockLimit())
                .fetch(candidateInfoJooqMapper::fromDb);
    }

    List<Long> getLockedWallets(int shard, long skipWorkerId, int limit) {
        return getLockedWallets(dslContextProvider.ppc(shard), skipWorkerId, limit);
    }

    List<Long> getLockedWallets(DSLContext dslContext, long skipWorkerId, int limit) {
        List<Long> result = dslContext
                .selectDistinct(WALLET_CID)
                .from(BS_EXPORT_QUEUE)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BS_EXPORT_QUEUE.CID))
                .where(BS_EXPORT_QUEUE.PAR_ID.isNotNull()
                        .and(BS_EXPORT_QUEUE.PAR_ID.ne(skipWorkerId))
                        .and(BS_EXPORT_QUEUE.PAR_ID.notIn(WORKER_IDS_NOT_RESTRICTED_BY_LOCKED_WALLETS))
                        .and(HAS_CAMPS_STAT)
                        .and(CAMPAIGNS.STATUS_BS_SYNCED.ne(CampaignsStatusbssynced.Yes))
                        .and(CAMPAIGNS.TYPE.eq(CampaignsType.wallet).or(CAMPAIGNS.WALLET_CID.gt(ID_NOT_SET)))
                )
                .limit(limit)
                .fetch(WALLET_CID);

        if (result.size() == limit) {
            logger.warn("getLockedWallets: exceeded limit {}", limit);
        }

        return result;
    }

    private static JooqMapperWithSupplier<BsExportCandidateInfo> createCandidateInfoMapper() {
        return createBaseMapperBuilder(BsExportCandidateInfo::new)
                .map(convertibleProperty(BsExportCandidateInfo.STATUS_BS_SYNCED, CAMPAIGNS.STATUS_BS_SYNCED,
                        CampaignMappings::statusBsSyncedFromDb,
                        CampaignMappings::statusBsSyncedToDb))
                .build();
    }
}
