package ru.yandex.direct.bsexport.logbroker;

import java.time.Duration;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import com.google.protobuf.InvalidProtocolBufferException;

import ru.yandex.direct.binlogbroker.logbroker_utils.writer.AbstractLogbrokerWriterImpl;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.BatchLogbrokerWriter;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogbrokerWriter;
import ru.yandex.direct.bsexport.messaging.FeedJsonSerializer;
import ru.yandex.direct.bsexport.model.FeedRequestMessage;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;

import static java.nio.charset.StandardCharsets.UTF_8;

@ParametersAreNonnullByDefault
public class FeedMessegesLogbrokerWriter implements LogbrokerWriter<FeedRequestMessage> {
    private final BatchLogbrokerWriter<FeedRequestMessage> batchLogbrokerWriter;
    private final int objectsInOneMessage;

    public FeedMessegesLogbrokerWriter(Supplier<CompletableFuture<AsyncProducer>> logbrokerProducerSupplier,
                                       Duration logbrokerTimeout,
                                       int retryCount, int objectsInOneMessage) {
        batchLogbrokerWriter = new BatchLogbrokerWriter<>(logbrokerProducerSupplier, logbrokerTimeout,
                retryCount, this::serializer);
        this.objectsInOneMessage = objectsInOneMessage;
    }

    AbstractLogbrokerWriterImpl.LogbrokerWriteRequest serializer(List<FeedRequestMessage> lst) {
        var printer = FeedJsonSerializer.getPrinter();
        StringJoiner serializedMessages = new StringJoiner(",", "[", "]");
        for (FeedRequestMessage feedRequestMessage : lst) {
            String serializedMessage;
            try {
                serializedMessage = printer.print(feedRequestMessage);
            } catch (InvalidProtocolBufferException e) {
                throw new FeedMessegesLogbrokerWriterException(getSerializeErrorMessage(feedRequestMessage), e);
            }
            serializedMessages.add(serializedMessage);
        }
        String jsonRequest = serializedMessages.toString();
        return batchLogbrokerWriter.new LogbrokerWriteRequest(jsonRequest.getBytes(UTF_8));
    }


    public CompletableFuture<Integer> write(List<FeedRequestMessage> data) {
        return batchLogbrokerWriter.write(Lists.partition(Objects.requireNonNull(data), objectsInOneMessage));
    }

    @Override
    public Long getInitialMaxSeqNo() {
        return batchLogbrokerWriter.getInitialMaxSeqNo();
    }

    @Override
    public void close() {
        batchLogbrokerWriter.close();
    }

    private String getSerializeErrorMessage(FeedRequestMessage feedRequestMessage) {
        return String.format("Failed to serialize message, uuid %s, campaignId %s, level %s",
                feedRequestMessage.getUuid(), feedRequestMessage.getCid(), feedRequestMessage.getLevel());
    }
}
