package ru.yandex.direct.bsexport.query.order;

import java.util.List;

import javax.annotation.Nullable;

import com.google.common.primitives.Ints;

import ru.yandex.direct.bsexport.exception.BsExportProcessingException;
import ru.yandex.direct.bsexport.model.BillingAggregateRule;
import ru.yandex.direct.bsexport.model.BillingAggregates;
import ru.yandex.direct.bsexport.snapshot.BsExportSnapshot;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.product.model.ProductType;

import static ru.yandex.direct.bsexport.model.ProductType.AudioCreativeReach;
import static ru.yandex.direct.bsexport.model.ProductType.VideoCreativeReach;
import static ru.yandex.direct.bsexport.model.ProductType.VideoCreativeReachIndoor;
import static ru.yandex.direct.bsexport.model.ProductType.VideoCreativeReachOutdoor;
import static ru.yandex.direct.bsexport.query.order.OrderDataFactory.underWallet;
import static ru.yandex.direct.core.entity.product.ProductConstants.PRODUCT_TYPE_BY_CAMPAIGN_TYPES;
import static ru.yandex.direct.core.entity.product.ProductConstants.getSpecialProductTypesByCampaignType;

class BillingAggregatesFactory {
    private final BsExportSnapshot snapshot;

    BillingAggregatesFactory(BsExportSnapshot snapshot) {
        this.snapshot = snapshot;
    }

    static List<ru.yandex.direct.bsexport.model.ProductType> productTypeMapper(ProductType productType) {
        switch (productType) {
            case CPM_VIDEO:
                return List.of(VideoCreativeReach);
            case CPM_OUTDOOR:
                return List.of(VideoCreativeReachOutdoor);
            case CPM_INDOOR:
                return List.of(VideoCreativeReachIndoor);
            case CPM_AUDIO:
                return List.of(AudioCreativeReach);
            default:
                throw new IllegalStateException("No mapped value for ProductType: " + productType);
        }
    }

    boolean canHaveBillingAggregates(CommonCampaign campaign) {
        if (!underWallet(campaign)) {
            return false;
        }

        return snapshot.strictlyGetWalletByCampaign(campaign).getIsSumAggregated();
    }

    /**
     * Получить биллинговые агрегаты для кампании.
     * <p>
     * Исходная реализация была в perl:
     * {@literal direct/perl/protected/BS/ExportQuery.pm?blame=true&rev=7112169#L7001}.
     *
     * @param campaign кампания
     * @return правила агрегации откруток или {@code null} (если не применимо)
     * @throws BsExportProcessingException при ошибке формирования объекта
     */
    @Nullable
    BillingAggregates getBillingAggregates(CommonCampaign campaign) {
        try {
            return internalGetBillingAggregates(campaign);
        } catch (IllegalArgumentException | IllegalStateException e) {
            throw new BsExportProcessingException("Failed to compose BillingAggregates object", e, campaign.getId());
        }
    }

    /**
     * @throws IllegalStateException    если для продукта нет правила агрегации
     * @throws IllegalArgumentException при числовом переполнении Id агрегата
     */
    @Nullable
    private BillingAggregates internalGetBillingAggregates(CommonCampaign campaign) {
        var defaultProductType = PRODUCT_TYPE_BY_CAMPAIGN_TYPES.get(campaign.getType());
        if (defaultProductType == null) {
            return null;
        }

        var defaultBillingAggregate = snapshot.getBillingAggregate(campaign.getWalletId(), defaultProductType);
        if (defaultBillingAggregate == null) {
            // если не найден агрегат по умолчанию, считаем, что для данного типа кампаний
            // пока не нужно отправлять никаких агрегатов
            return null;
        }

        var builder = BillingAggregates.newBuilder();
        int defaultBillingAggregateId = Ints.checkedCast(defaultBillingAggregate.getId());
        builder.setDefault(defaultBillingAggregateId);

        var specialProductTypesByCampaignType = getSpecialProductTypesByCampaignType(campaign.getCurrency());
        var specialsProductTypes = specialProductTypesByCampaignType.getOrDefault(campaign.getType(), List.of());
        for (var specialProductType : specialsProductTypes) {
            var rule = getBillingAggregateRule(campaign.getWalletId(), specialProductType);
            if (rule != null) {
                builder.addRules(rule);
            }
        }

        return builder.build();
    }

    @Nullable
    private BillingAggregateRule getBillingAggregateRule(Long walletId, ProductType specialProductType) {
        var billingAggregate = snapshot.getBillingAggregate(walletId, specialProductType);
        if (billingAggregate == null) {
            // для такого типа продукта нет агрегатов
            return null;
        }

        int billingAggregateId = Ints.checkedCast(billingAggregate.getId());
        var productTypes = productTypeMapper(specialProductType);

        return BillingAggregateRule.newBuilder()
                .setResult(billingAggregateId)
                .addAllProductTypes(productTypes)
                .build();
    }
}
