package ru.yandex.direct.bsexport.query.order;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bsexport.iteration.container.ExportBatchLimits;
import ru.yandex.direct.bsexport.model.AttributionType;
import ru.yandex.direct.bsexport.model.ContentType;
import ru.yandex.direct.bsexport.model.Order;
import ru.yandex.direct.bsexport.model.OrderType;
import ru.yandex.direct.bsexport.snapshot.BsExportSnapshot;
import ru.yandex.direct.bsexport.snapshot.model.ExportedCampaign;
import ru.yandex.direct.bsexport.snapshot.model.ExportedClient;
import ru.yandex.direct.bsexport.snapshot.model.ExportedUser;
import ru.yandex.direct.bsexport.snapshot.model.QueuedCampaign;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignAttributionModel;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithAttributionModel;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithEshowsSettings;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithImpressionStandardTime;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithNetworkSettings;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithWalletId;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.InternalCampaign;
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants;
import ru.yandex.direct.utils.CommonUtils;

import static ru.yandex.direct.bsexport.query.QueryUtil.FALSE;
import static ru.yandex.direct.bsexport.query.QueryUtil.booleanToInt;
import static ru.yandex.direct.bsexport.query.order.EshowsVideoTypeFactory.getEshowsVideoType;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.CPM_DEALS;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.CPM_PRICE;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.PERFORMANCE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_ATTRIBUTION_MODEL;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DISALLOW_CAMPAIGN_NAME_LETTERS_RE;
import static ru.yandex.direct.core.entity.client.model.ClientFlags.AS_SOON_AS_POSSIBLE;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;

@ParametersAreNonnullByDefault
public class OrderDataFactory {
    private static final Logger logger = LoggerFactory.getLogger(OrderDataFactory.class);
    /**
     * AgencyID Бегун'а. Получился из старого спискаа по принзнаку is_begun.
     * https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/API/Settings.pm?rev=6219335#L187
     */
    private static final Set<Long> BEGUN_AGENCY_IDS = Set.of(
            1647047L,   // begun-2011
            1618235L    // begun-test-agency
    );
    private static final long TEST_BEGUN_CAMPAIGN_ID = 2834265L;

    private static final DateTimeFormatter QUEUE_SET_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");

    private static final DateTimeFormatter CAMPAIGN_START_TIME_FORMATTER = DateTimeFormatter.ofPattern(
            "yyyyMMdd000000");
    private ExportBatchLimits exportBatchLimits;

    private final Map<Long, Order.Builder> orders;
    private final BsExportSnapshot snapshot;
    private final BillingAggregatesFactory billingAggregatesFactory;


    public OrderDataFactory(BsExportSnapshot snapshot) {
        this.snapshot = snapshot;

        orders = new HashMap<>();
        billingAggregatesFactory = new BillingAggregatesFactory(snapshot);
    }

    public Order.Builder getOrder(Long campaignId) {
        return orders.computeIfAbsent(campaignId, this::createBaseOrder);
    }

    private Order.Builder createBaseOrder(Long campaignId) {
        CommonCampaign campaign = snapshot.strictlyGetCampaign(campaignId);
        var client = snapshot.strictlyGetClientByCampaign(campaign);
        Order.Builder builder = Order.newBuilder()
                .setEID(campaignId)
                .setID(campaign.getOrderId()) //TODO заготавливать в снепшоте значение для новых кампаний
                .setUpdateInfo(FALSE)
                .setStop(getStop(campaign))
                .setOrderType(getOrderType(campaign))
                .setGroupOrder(getGroupOrder(campaign))
                .setGroupOrderID(getGroupOrderId(campaign))
                .setClientID(campaign.getClientId())
                .setAgencyID(getAgencyId(campaign))
                .setManagerUID(getManagerUid(campaign))
                .setContentType(getContentType(campaign))
                .setIndependentBids(getIndependentBids(campaign))
                .setQueueSetTime(getQueueSetTime(campaign))
                .setDescription(getDescription(campaign))
                .setAttributionType(getAttributionType(campaign))
                .setStartTime(getStartTime(campaign))
                .setContextPriceCoef(getContextPriceCoef(campaign));

        addIsPriorityCampaign(builder, client);
        // кампании с типом сделки пока не перенесены в java
        addPrivateDeal(builder, campaign);
        addAuctionPriority(builder, campaign);
        addOrderCreateTime(builder, campaign);
        addInternalCampaignFields(builder, campaign);
        addBillingOrders(builder, campaign);
        addFieldsFromCampaignFlags(builder, campaign);
        addImpressionStandardTimeFields(builder, campaign);
        addEshowsFields(builder, campaign);
        // Помечено deprecated так как значение в базе не заполняются ниоткуда, кроме копирования со старых кампаний.
        // Нужно аккуратно отпилить это поле из экспорта совсем, или всегда посылать умолчание.
        // noinspection deprecation
        builder.setAutoOptimization(getAutoOptimization(campaign));

        // устарело в DIRECT-20562: Отключить понижающие "антипохиленко" коэффициенты
        //noinspection deprecation
        builder.setMaxCPC(0);

        return builder;
    }

    int getOrderType(CommonCampaign campaign) {
        // Первоначальная реализация DIRECT-14138: Автоматическая пометка внутренней рекламы
        CampaignType campaignType = campaign.getType();
        if (campaignType == CampaignType.GEO) {
            return OrderType.geocontext_VALUE;
        }

        ExportedClient exportedClient = snapshot.strictlyGetClientByCampaign(campaign);
        if (exportedClient.getIsBusinessUnit()) {
            // DIRECT-54147: Новый OrderType для кампаний БЮ
            return OrderType.intercompany_VALUE;
        }

        ExportedUser exportedUser = snapshot.strictlyGetUserByCampaign(campaign);
        if (CampaignTypeKinds.INTERNAL.contains(campaignType)
                || exportedUser.getStatusYandexAdv()
                // DIRECT-14138: Автоматическая пометка внутренней рекламы
                || campaign.getPaidByCertificate()
                // DIRECT-36045: Передвать OrderType=6 для кампаний под ОС, если ОС был оплачен сертификатом
                || hasWalletPaidByCertificate(campaign)
        ) {
            return OrderType.own_VALUE;
        }

        if (campaign.getAgencyId() != null && BEGUN_AGENCY_IDS.contains(campaign.getAgencyId())
                || campaign.getId() == TEST_BEGUN_CAMPAIGN_ID
        ) {
            // DIRECT-12158: API: новый метод для получения статистики
            return OrderType.begun_VALUE;
        }

        return OrderType.commercial_VALUE;
    }

    private boolean hasWalletPaidByCertificate(CommonCampaign campaign) {
        if (isNotUnderWallet(campaign)) {
            return false;
        }
        WalletTypedCampaign walletCampaign = snapshot.strictlyGetWalletByCampaign(campaign);
        return walletCampaign.getPaidByCertificate();
    }

    long getGroupOrderId(CommonCampaign campaign) {
        if (isNotUnderWallet(campaign)) {
            return 0L;
        }
        WalletTypedCampaign walletCampaign = snapshot.strictlyGetWalletByCampaign(campaign);
        return walletCampaign.getOrderId();
    }

    String getQueueSetTime(BaseCampaign campaign) {
        Long campaignId = campaign.getId();
        QueuedCampaign queuedCampaign = snapshot.strictlyGetQueuedCampaign(campaignId);
        LocalDateTime queueTime = queuedCampaign.getQueueTime();
        return queueTime.format(QUEUE_SET_TIME_FORMATTER);
    }

    int getAutoOptimization(BaseCampaign campaign) {
        Long campaignId = campaign.getId();
        ExportedCampaign exportedCampaign = snapshot.strictlyGetExportedCampaign(campaignId);
        boolean autoOptimization = exportedCampaign.getAutoOptimization();
        return booleanToInt(autoOptimization);
    }


    static int getStop(CommonCampaign campaign) {
        // TODO DIRECT-114417: поддержать остановку cpm_price кампаний по корректности и статусу
        Boolean statusShow = campaign.getStatusShow();
        boolean stop = !statusShow;
        return booleanToInt(stop);
    }

    static int getAttributionType(CommonCampaign campaign) {
        CampaignAttributionModel attributionModel;
        if (campaign instanceof CampaignWithAttributionModel) {
            attributionModel = ((CampaignWithAttributionModel) campaign).getAttributionModel();
        } else {
            attributionModel = DEFAULT_ATTRIBUTION_MODEL;
        }
        return attributionTypeMapper(attributionModel);
    }

    static int attributionTypeMapper(CampaignAttributionModel attributionModel) {
        switch (attributionModel) {
            case LAST_CLICK:
                return AttributionType.LAST_CLICK_VALUE;
            case LAST_SIGNIFICANT_CLICK:
                return AttributionType.LAST_SIGNIFICANT_CLICK_VALUE;
            case FIRST_CLICK:
                return AttributionType.FIRST_CLICK_VALUE;
            case LAST_YANDEX_DIRECT_CLICK:
                return AttributionType.LAST_YANDEX_DIRECT_CLICK_VALUE;
            case LAST_SIGNIFICANT_CLICK_CROSS_DEVICE:
                return AttributionType.LAST_SIGNIFICANT_CLICK_CROSS_DEVICE_VALUE;
            case FIRST_CLICK_CROSS_DEVICE:
                return AttributionType.FIRST_CLICK_CROSS_DEVICE_VALUE;
            case LAST_YANDEX_DIRECT_CLICK_CROSS_DEVICE:
                return AttributionType.LAST_YANDEX_DIRECT_CLICK_CROSS_DEVICE_VALUE;
            default:
                throw new IllegalStateException("No mapped value for CampaignAttributionModel: " + attributionModel);
        }
    }

    /**
     * Использовать биллинговые агрегаты для откруток из этой кампании, если кампания под ОС, и этот ОС
     * перешел на схему учета зачислений, когда открученные зачисления не переносятся на дочерние кампании, а
     * остаются на ОС.
     */
    void addBillingOrders(Order.Builder builder, CommonCampaign campaign) {
        if (!billingAggregatesFactory.canHaveBillingAggregates(campaign)) {
            return;
        }

        var billingAggregates = billingAggregatesFactory.getBillingAggregates(campaign);
        if (billingAggregates == null) {
            return;
        }

        builder.setBillingOrders(billingAggregates);
    }

    void addOrderCreateTime(Order.Builder builder, CommonCampaign campaign) {
        if (Objects.nonNull(campaign.getCreateTime())) {
            builder.setOrderCreateTime(campaign.getCreateTime().atZone(MSK).toEpochSecond());
        }
    }

    static int getGroupOrder(CampaignWithCampaignType campaign) {
        boolean isWallet = campaign.getType() == CampaignType.WALLET;
        return booleanToInt(isWallet);
    }

    static ContentType getContentType(CampaignWithCampaignType campaign) {
        return contentTypeMapper(campaign.getType());
    }

    int getIsVirtual(CommonCampaign campaign) {
        return booleanToInt(campaign.getIsVirtual());
    }

    String getStartTime(CommonCampaign campaign) {
        return CAMPAIGN_START_TIME_FORMATTER.format(campaign.getStartDate());
    }

    int getContextPriceCoef(CommonCampaign campaign) {
        // todo юнит тест, что тип mobile_content реализуется с этим полем
        if (campaign instanceof CampaignWithNetworkSettings) {
            var textCamp = (CampaignWithNetworkSettings) campaign;
            return textCamp.getContextPriceCoef();
        }
        return CampaignConstants.DEFAULT_CONTEXT_LIMIT;
    }

    void addInternalCampaignFields(Order.Builder builder, CommonCampaign campaign) {
        if (campaign instanceof InternalCampaign) {
            var internalCampaign = (InternalCampaign) campaign;
            builder.setPlaceID(internalCampaign.getPlaceId());
            var internalAdsProduct = snapshot.getInternalAdsProduct(campaign.getClientId());
            if (Objects.nonNull(internalAdsProduct)) {
                builder.setServiceName(internalAdsProduct.getName());
            }
        }
    }

    void addEshowsFields(Order.Builder builder, CommonCampaign campaign) {
        if (campaign instanceof CampaignWithEshowsSettings) {
            var eshowsSettings = ((CampaignWithEshowsSettings) campaign).getEshowsSettings();
            if (Objects.nonNull(eshowsSettings.getBannerRate())) {
                builder.setEshowsBannerRate(truncate(eshowsSettings.getBannerRate().getTypedValue(), 2));
            }
            if (Objects.nonNull(eshowsSettings.getVideoRate())) {
                builder.setEshowsVideoRate(truncate(eshowsSettings.getVideoRate().getTypedValue(), 2));
            }
            if (Objects.nonNull(eshowsSettings.getVideoType())) {
                var exportedEshowsVideoType = getEshowsVideoType(eshowsSettings.getVideoType());
                if (Objects.isNull(exportedEshowsVideoType)) {
                    throw new IllegalStateException("Invalid eshows video type: " + eshowsSettings.getVideoType() +
                            ", cid " + campaign.getId());
                } else {
                    builder.setEshowsVideoType(exportedEshowsVideoType);
                }
            }
        }
    }

    void addImpressionStandardTimeFields(Order.Builder builder, CommonCampaign campaign) {
        if (campaign instanceof CampaignWithImpressionStandardTime) {
            var time =
                    ((CampaignWithImpressionStandardTime) campaign).getImpressionStandardTime();
            if (Objects.nonNull(time)) {
                var type = ImpressionStandardTypeFactory.getImpresstionStandardType(time);
                builder.setImpressionStandardTime(time.getTypedValue().intValue());
                builder.setImpressionStandardType(type);
            }
        }
    }

    Double truncate(Double sourceValue, int scale) {
        BigDecimal bigDecimal = BigDecimal.valueOf(sourceValue);
        return bigDecimal.setScale(scale, RoundingMode.DOWN).doubleValue();
    }

    void addFieldsFromCampaignFlags(Order.Builder builder, CommonCampaign campaign) {
        builder.setIsVirtual(getIsVirtual(campaign));
        if (campaign.getIsAloneTrafaretAllowed()) {
            builder.setAllowAloneTrafaret(1);
        }
        if (campaign.getRequireFiltrationByDontShowDomains()) {
            builder.setRequireFiltrationByDontShowDomains(1);
        }

        if (campaign.getHasTurboApp()) {
            builder.setHasTurboApp(true);
        }

        if (PERFORMANCE.equals(campaign.getType())) {
            builder.setUseTurboUrl(booleanToInt(campaign.getHasTurboSmarts()));
        }
    }

    void addIsPriorityCampaign(Order.Builder builder, ExportedClient client) {
        if (client.getFlags().contains(AS_SOON_AS_POSSIBLE)) {
            builder.setIsPriorityCampaign(1);
        }
    }

    void addPrivateDeal(Order.Builder builder, CommonCampaign campaign) {
        if (CPM_DEALS.equals(campaign.getType())) {
            builder.setPrivateDeal(1);
        }
    }

    void addAuctionPriority(Order.Builder builder, CommonCampaign campaign) {
        if (CPM_PRICE.equals(campaign.getType())) {
            builder.setAuctionPriority(10);
        }
    }

    /**
     * Получить тип заказа.
     * У большинства соответствует Директовому, у охватных - отдельный.
     * <p>
     * Пояснение к исключениям (отсутствующим типам):
     * <ul>
     * <li>МКБ раньше отправлялся отдельным транспортом (ныне выпелен)</li>
     * <li>Биллинговый агрегат - это не самостоятельная кампания, в транспорте не отправляются как Order</li>
     * </ul>
     */
    static ContentType contentTypeMapper(CampaignType campaignType) {
        switch (campaignType) {
            case TEXT:
            case CONTENT_PROMOTION:         // https://st.yandex-team.ru/DIRECT-90956#5cadb0a73cbb5c001f3d87e8
                return ContentType.text;

            case GEO:
                //noinspection deprecation
                return ContentType.geo;

            case WALLET:
                return ContentType.wallet;

            case DYNAMIC:
                return ContentType.dynamic;

            case MOBILE_CONTENT:
                return ContentType.mobile_content;

            case PERFORMANCE:
                return ContentType.performance;

            case MCBANNER:
                return ContentType.mcbanner;

            case CPM_BANNER:
            case CPM_DEALS:
            case CPM_YNDX_FRONTPAGE:
            case CPM_PRICE:
                return ContentType.reach;

            case INTERNAL_DISTRIB:
                return ContentType.internal_distrib;

            case INTERNAL_FREE:
                return ContentType.internal_free;

            case INTERNAL_AUTOBUDGET:
                return ContentType.internal_autobudget;

            default:
                throw new IllegalStateException("No mapped value for CampaignType: " + campaignType);

        }
    }

    static int getIndependentBids(CommonCampaign campaign) {
        boolean independentBids = false;
        if (campaign instanceof CampaignWithStrategy) {
            independentBids = ((CampaignWithStrategy) campaign).getStrategy().isDifferentPlaces();
        }
        return booleanToInt(independentBids);
    }

    static long getManagerUid(CommonCampaign campaign) {
        Long managerUid = campaign.getManagerUid();
        if (managerUid == null) {
            return 0L;
        }
        return managerUid;
    }

    static String getDescription(CommonCampaign campaign) {
        Long campaignId = campaign.getId();
        // Удаляем все "необычные" символы, чтобы не ломать крутилку и внешних потребителей.
        // DIRECT-60507: выкидывать при отправке в БК из названий кампании все символы кроме ALLOW_BANNER_LETTER
        String name = cleanCampaignName(campaign.getName());
        return campaignId + ": " + name;
    }

    private static String cleanCampaignName(String name) {
        return name.replaceAll(DISALLOW_CAMPAIGN_NAME_LETTERS_RE, "");
    }

    private static long getAgencyId(CommonCampaign campaign) {
        Long agencyId = campaign.getAgencyId();
        if (agencyId == null) {
            return 0L;
        }
        return agencyId;
    }

    private static boolean isNotUnderWallet(CampaignWithWalletId campaign) {
        Long walletId = campaign.getWalletId();
        return walletId == null || walletId == 0L;
    }

    static boolean underWallet(CampaignWithWalletId campaign) {
        return CommonUtils.isValidId(campaign.getWalletId());
    }
}

