package ru.yandex.direct.bsexport.snapshot;

import java.util.Collection;
import java.util.List;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bsexport.exception.BsExportProcessingException;
import ru.yandex.direct.bsexport.exception.ObjectNotFoundInSnapshotException;
import ru.yandex.direct.bsexport.iteration.BsExportIterationContext;
import ru.yandex.direct.bsexport.snapshot.holders.BillingAggregateRequest;
import ru.yandex.direct.bsexport.snapshot.holders.BillingAggregatesHolder;
import ru.yandex.direct.bsexport.snapshot.holders.BrandsafetyRetargetingConditionsHolder;
import ru.yandex.direct.bsexport.snapshot.holders.CampaignsHolder;
import ru.yandex.direct.bsexport.snapshot.holders.ClientsHolder;
import ru.yandex.direct.bsexport.snapshot.holders.CryptaGoalsHolder;
import ru.yandex.direct.bsexport.snapshot.holders.ExportedCampaignsHolder;
import ru.yandex.direct.bsexport.snapshot.holders.InternalAdsProductsHolder;
import ru.yandex.direct.bsexport.snapshot.holders.QueuedCampaignsHolder;
import ru.yandex.direct.bsexport.snapshot.holders.UsersHolder;
import ru.yandex.direct.bsexport.snapshot.internal.FirstPassDataFetcher;
import ru.yandex.direct.bsexport.snapshot.model.ExportedCampaign;
import ru.yandex.direct.bsexport.snapshot.model.ExportedClient;
import ru.yandex.direct.bsexport.snapshot.model.ExportedUser;
import ru.yandex.direct.bsexport.snapshot.model.QueuedCampaign;
import ru.yandex.direct.core.entity.campaign.model.BillingAggregateCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithWalletId;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProduct;
import ru.yandex.direct.core.entity.product.model.ProductType;

@ParametersAreNonnullByDefault
public class BsExportSnapshot {
    private static final Logger logger = LoggerFactory.getLogger(BsExportSnapshot.class);

    private final QueuedCampaignsFilteringStep queuedCampaignsFilteringStep;
    private final QueuedCampaignsHolder queuedCampaignsHolder;
    private final CampaignsHolder campaignsHolder;
    private final ExportedCampaignsHolder exportedCampaignsHolder;
    private final UsersHolder usersHolder;
    private final ClientsHolder clientsHolder;
    private final InternalAdsProductsHolder internalAdsProductsHolder;
    private final BillingAggregatesHolder billingAggregatesHolder;
    private final BrandsafetyRetargetingConditionsHolder brandsafetyRetargetingConditionsHolder;
    private final CryptaGoalsHolder cryptaGoalsHolder;

    public BsExportSnapshot(SnapshotDataFactory factory, BsExportIterationContext iterationContext) {
        Supplier<Collection<Long>> idsToFetchQueuedCampaigns = iterationContext::getDataCampaignIds;
        queuedCampaignsFilteringStep = new QueuedCampaignsFilteringStep(iterationContext,
                idsToFetchQueuedCampaigns, factory.campaignTypedRepository::getSupportedTypes);
        queuedCampaignsHolder = new QueuedCampaignsHolder(factory.bsExportRepository, queuedCampaignsFilteringStep,
                idsToFetchQueuedCampaigns, iterationContext.getBatchLimits()::getDbRowsLimit);

        Supplier<Collection<Long>> idsToFetchCampaignsData = queuedCampaignsHolder::getCampaignAndWalletIds;
        campaignsHolder = new CampaignsHolder(factory.campaignTypedRepository, idsToFetchCampaignsData);
        exportedCampaignsHolder = new ExportedCampaignsHolder(factory.bsExportRepository, idsToFetchCampaignsData);

        usersHolder = new UsersHolder(factory.bsExportRepository, campaignsHolder::getUserIds);
        clientsHolder = new ClientsHolder(factory.bsExportRepository, campaignsHolder::getClientIds);
        internalAdsProductsHolder = new InternalAdsProductsHolder(factory.internalAdsProductRepository,
                campaignsHolder::getCampaigns);
        billingAggregatesHolder = new BillingAggregatesHolder(factory.campaignTypedRepository,
                factory.productRepository, clientsHolder::getClientIds);
        brandsafetyRetargetingConditionsHolder =
                new BrandsafetyRetargetingConditionsHolder(factory.retargetingConditionRepository,
                campaignsHolder::getCampaigns);
        cryptaGoalsHolder = new CryptaGoalsHolder(factory.cryptaSegmentRepository,
                brandsafetyRetargetingConditionsHolder::getGoalIds);
    }

    private BsExportSnapshot(Builder builder) {
        queuedCampaignsFilteringStep = builder.preFilterCampaignsStep;
        queuedCampaignsHolder = builder.queuedCampaignsHolder;
        campaignsHolder = builder.campaignsHolder;
        exportedCampaignsHolder = builder.exportedCampaignsHolder;
        usersHolder = builder.usersHolder;
        clientsHolder = builder.clientsHolder;
        internalAdsProductsHolder = builder.internalAdsProductsHolder;
        billingAggregatesHolder = builder.billingAggregatesHolder;
        brandsafetyRetargetingConditionsHolder = builder.brandsafetyRetargetingConditionsHolder;
        cryptaGoalsHolder = builder.cryptaGoalsHolder;
    }


    static class Builder {
        private QueuedCampaignsFilteringStep preFilterCampaignsStep;
        private QueuedCampaignsHolder queuedCampaignsHolder;
        private CampaignsHolder campaignsHolder;
        private ExportedCampaignsHolder exportedCampaignsHolder;
        private UsersHolder usersHolder;
        private ClientsHolder clientsHolder;
        private InternalAdsProductsHolder internalAdsProductsHolder;
        private BillingAggregatesHolder billingAggregatesHolder;
        private BrandsafetyRetargetingConditionsHolder brandsafetyRetargetingConditionsHolder;
        private CryptaGoalsHolder cryptaGoalsHolder;

        /**
         * Конструктор снепшота для тестов.
         * Без публичного доступа, чтобы не использовать в продакшн-коде.
         */
        Builder() {
        }

        Builder withQueuedCampaignsHolder(QueuedCampaignsHolder queuedCampaignsHolder) {
            this.queuedCampaignsHolder = queuedCampaignsHolder;
            return this;
        }

        Builder withCampaignsHolder(CampaignsHolder campaignsHolder) {
            this.campaignsHolder = campaignsHolder;
            return this;
        }

        Builder withExportedCampaignsHolder(ExportedCampaignsHolder exportedCampaignsHolder) {
            this.exportedCampaignsHolder = exportedCampaignsHolder;
            return this;
        }

        Builder withUsersHolder(UsersHolder usersHolder) {
            this.usersHolder = usersHolder;
            return this;
        }

        Builder withClientsHolder(ClientsHolder clientsHolder) {
            this.clientsHolder = clientsHolder;
            return this;
        }

        Builder withInternalAdsProductsHolder(InternalAdsProductsHolder internalAdsProductsHolder) {
            this.internalAdsProductsHolder = internalAdsProductsHolder;
            return this;
        }

        Builder withBillingAggregatesHolder(BillingAggregatesHolder billingAggregatesHolder) {
            this.billingAggregatesHolder = billingAggregatesHolder;
            return this;
        }

        Builder withBrandSafetyRetConditionsHolder(BrandsafetyRetargetingConditionsHolder brandsafetyRetargetingConditionsHolder) {
            this.brandsafetyRetargetingConditionsHolder = brandsafetyRetargetingConditionsHolder;
            return this;
        }

        Builder withCryptaGoalsHolder(CryptaGoalsHolder cryptaGoalsHolder) {
            this.cryptaGoalsHolder = cryptaGoalsHolder;
            return this;
        }

        BsExportSnapshot buildForTests() {
            return new BsExportSnapshot(this);
        }
    }

    /**
     * Получить список "получателей" (из базы) данных снепшота)
     *
     * @implNote порядок важен: фетчеры могут загружать новые данные, основываясь на уже загруженных,
     * например получать данные о клиентах по кампаниям
     */
    List<FirstPassDataFetcher> getFirstPassFetchers() {
        return List.of(
                queuedCampaignsHolder,
                queuedCampaignsFilteringStep,
                campaignsHolder,
                exportedCampaignsHolder,
                usersHolder,
                clientsHolder,
                internalAdsProductsHolder,
                billingAggregatesHolder,
                brandsafetyRetargetingConditionsHolder,
                cryptaGoalsHolder
        );
    }

    void preFetch(DSLContext dslContext) {
        // TODO DIRECT-112266: выбирать к отправке id групп и баннеров
    }

    /**
     * Получить данные о кампании в очереди из снепшота
     *
     * @param campaignId id кампании
     * @throws ObjectNotFoundInSnapshotException если кампании не нашлось
     */
    public QueuedCampaign strictlyGetQueuedCampaign(Long campaignId) {
        var campaign = queuedCampaignsHolder.getObject(campaignId);
        if (campaign == null) {
            throw new ObjectNotFoundInSnapshotException(campaignId, "QueuedCampaign", "id", campaignId);
        }
        return campaign;
    }

    /**
     * Получить кампанию из снепшота
     *
     * @param campaignId id кампании
     * @throws ObjectNotFoundInSnapshotException если кампании не нашлось
     */
    public CommonCampaign strictlyGetCampaign(Long campaignId) {
        var campaign = campaignsHolder.getObject(campaignId);
        if (campaign == null) {
            throw new ObjectNotFoundInSnapshotException(campaignId, "Campaign", "id", campaignId);
        }
        return campaign;
    }

    /**
     * Получить продукт внутренней кампании
     *
     * @param clientId id клиента
     */
    @Nullable
    public InternalAdsProduct getInternalAdsProduct(Long clientId) {
        return internalAdsProductsHolder.getObject(clientId);

    }

    /**
     * Получить биллинговые аггрегаты для кошелька и продукта
     *
     * @param walletId    id кошелька
     * @param productType тип продукта кампании
     */
    @Nullable
    public BillingAggregateCampaign getBillingAggregate(Long walletId, ProductType productType) {
        return billingAggregatesHolder.getObject(new BillingAggregateRequest(walletId, productType));

    }

    /**
     * Получить дополнительные данные кампании для экспорта
     *
     * @param campaignId id кампании
     * @throws ObjectNotFoundInSnapshotException если кампании не нашлось
     */
    public ExportedCampaign strictlyGetExportedCampaign(Long campaignId) {
        var campaign = exportedCampaignsHolder.getObject(campaignId);
        if (campaign == null) {
            throw new ObjectNotFoundInSnapshotException(campaignId, "ExportedCampaign", "id", campaignId);
        }
        return campaign;
    }

    /**
     * Получить данные клиента из снепшота по {@code ClientID} из кампании
     *
     * @param campaign кампания, по которой получаем данные
     * @throws ObjectNotFoundInSnapshotException если клиента в снепшоте не нашлось
     */
    public ExportedClient strictlyGetClientByCampaign(CommonCampaign campaign) {
        Long clientId = campaign.getClientId();
        ExportedClient client = clientsHolder.getObject(clientId);
        if (client == null) {
            throw new ObjectNotFoundInSnapshotException(campaign.getId(), "Client", "id", clientId);
        }
        return client;
    }

    /**
     * Получить данные пользователя из снепшота по {@code uid} из кампании
     *
     * @param campaign кампания, по которой получаем данные
     * @throws ObjectNotFoundInSnapshotException если пользователя в снепшоте не нашлось
     */
    public ExportedUser strictlyGetUserByCampaign(CommonCampaign campaign) {
        Long uid = campaign.getUid();
        ExportedUser user = usersHolder.getObject(uid);
        if (user == null) {
            throw new ObjectNotFoundInSnapshotException(campaign.getId(), "User", "uid", uid);
        }
        return user;
    }

    /**
     * Получить данные кампании-кошелька по кампании
     *
     * @param campaign кампания, по которой получаем данные
     * @throws ObjectNotFoundInSnapshotException если кампании-кошелька нет в снепшоте (или у кампании нет кошелька!)
     * @throws BsExportProcessingException       если кампания-кошелек почему-то не кошелек
     */
    public WalletTypedCampaign strictlyGetWalletByCampaign(CampaignWithWalletId campaign) {
        Long walletId = campaign.getWalletId();
        var walletCampaign = campaignsHolder.getObject(walletId);
        if (walletCampaign == null) {
            throw new ObjectNotFoundInSnapshotException(campaign.getId(), "Wallet", "id", walletId);
        }
        if (!(walletCampaign instanceof WalletTypedCampaign)) {
            String message = String.format("Unexpected type for walletId %d: %s",
                    walletId, walletCampaign.getClass().getCanonicalName());
            throw new BsExportProcessingException(message, campaign.getId());
        }
        return (WalletTypedCampaign) walletCampaign;
    }
}
