package ru.yandex.direct.bsexport.snapshot;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bsexport.iteration.BsExportIterationContext;
import ru.yandex.direct.bsexport.snapshot.internal.FirstPassDataFetcher;
import ru.yandex.direct.bsexport.snapshot.model.QueuedCampaign;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;

import static com.google.common.base.Preconditions.checkState;

/**
 * Класс отвечает за: <ul>
 * <li>Разблокировку кампаний, не выбранных из базы по лимитам на число строк</li>
 * <li>Разблокироку кампаний, типы которых пока не поддержвается (в репозитории кампаний на интерфейсах)</li>
 * <li>Отбор кампаний, которые не должны быть отправлены (в perl это фильтровалось на уровне sql)</li>
 * </ul>
 */
@ParametersAreNonnullByDefault
public class QueuedCampaignsFilteringStep implements FirstPassDataFetcher {
    private static final Logger logger = LoggerFactory.getLogger(QueuedCampaignsFilteringStep.class);

    private static final List<Predicate<QueuedCampaign>> CHECKS = List.of(
            QueuedCampaignsFilteringStep::isNotEmpty,
            QueuedCampaignsFilteringStep::wasInBsOrAcceptedOnModeration,
            QueuedCampaignsFilteringStep::hasMoneyOrNeedSync,
            // TODO DIRECT-120174: реализовать отправку новых кампаний (без orderid)
            QueuedCampaignsFilteringStep::wasInBs
    );

    private final BsExportIterationContext iterationContext;
    private final Supplier<Collection<Long>> campaignIdsSupplier;
    private final Supplier<Collection<CampaignType>> campaignTypesSupplier;
    private final Set<Long> campaignIdsForUnlock;
    private final List<Long> notEligibleCampaignIds;

    private boolean collectedExcess;
    private boolean collectedUnsupported;
    private boolean collectedNotEligible;


    QueuedCampaignsFilteringStep(BsExportIterationContext iterationContext,
                                 Supplier<Collection<Long>> campaignIdsSupplier,
                                 Supplier<Collection<CampaignType>> campaignTypesSupplier) {
        this.iterationContext = iterationContext;
        this.campaignIdsSupplier = campaignIdsSupplier;
        this.campaignTypesSupplier = campaignTypesSupplier;

        campaignIdsForUnlock = new HashSet<>();
        notEligibleCampaignIds = new ArrayList<>();
    }

    @Override
    public void firstFetch(DSLContext ignored) {
        checkState(collectedExcess);
        checkState(collectedUnsupported);
        checkState(collectedNotEligible);

        if (!campaignIdsForUnlock.isEmpty()) {
            iterationContext.unlockCampaigns(campaignIdsForUnlock);
        }
        if (!notEligibleCampaignIds.isEmpty()) {
            // TODO DIRECT-114931: что-то делать с кампаниями, не проходящими по sql-условиям к отправке
        }
    }

    /**
     * Получить id кампаний, которые хотели загрузить, но они не не попали в выборку
     *
     * @param queuedCampaignIds идентификаторы кампаний выбранных из базы
     */
    public List<Long> collectExcessCampaignIds(Set<Long> queuedCampaignIds) {
        List<Long> result = new ArrayList<>();
        for (Long campaignId : campaignIdsSupplier.get()) {
            if (queuedCampaignIds.contains(campaignId)) {
                continue;
            }
            campaignIdsForUnlock.add(campaignId);
            result.add(campaignId);
        }
        collectedExcess = true;
        return result;
    }

    /**
     * Получить id кампаний, тип которых пока не поддерживается (для дальнейшей обработки)
     * х
     *
     * @param queuedCampaigns выбранные из базы кампании
     */
    public List<Long> collectUnsupportedCampaignIds(Collection<QueuedCampaign> queuedCampaigns) {
        Collection<CampaignType> supportedTypes = campaignTypesSupplier.get();
        List<Long> result = new ArrayList<>();
        for (QueuedCampaign queuedCampaign : queuedCampaigns) {
            if (supportedTypes.contains(queuedCampaign.getType())) {
                continue;
            }
            Long campaignId = queuedCampaign.getId();
            campaignIdsForUnlock.add(campaignId);
            result.add(campaignId);
        }
        collectedUnsupported = true;
        return result;
    }

    /**
     * Получить id кампаний, которые не пригодны к отправке
     *
     * @param queuedCampaigns выбранные из базы кампании
     */
    public List<Long> collectNotEligibleCampaignIds(Collection<QueuedCampaign> queuedCampaigns) {
        List<Long> result = new ArrayList<>();
        for (QueuedCampaign queuedCampaign : queuedCampaigns) {
            if (isEligible(queuedCampaign)) {
                continue;
            }
            Long campaignId = queuedCampaign.getId();
            notEligibleCampaignIds.add(campaignId);
            result.add(campaignId);
        }
        collectedNotEligible = true;
        return result;
    }

    private static boolean isEligible(QueuedCampaign queuedCampaign) {
        return CHECKS.stream()
                .map(check -> check.test(queuedCampaign))
                .reduce(Boolean.TRUE, Boolean::logicalAnd);
    }

    private static boolean isNotEmpty(QueuedCampaign info) {
        if (!info.getStatusEmpty()) {
            return true;
        }
        logger.trace("Filtered campaign {}: is empty", info.getId());
        return false;
    }

    private static boolean wasInBsOrAcceptedOnModeration(QueuedCampaign info) {
        if (info.getStatusModerate() == CampaignStatusModerate.YES
                || wasInBs(info)
        ) {
            return true;
        }
        logger.trace("Filtered campaign {}: unmoderated", info.getId());
        return false;
    }


    private static boolean hasMoneyOrNeedSync(QueuedCampaign info) {
        // TODO: помотреть, в каких случаях это срабатывает и разобрать, унеся проверки в код формирования запроса
        // может быть оставить проверку "OrderID или деньги"

        if (moneyCondition(info) && info.getStatusPostModerate() == CampaignStatusPostmoderate.ACCEPTED) {
            return true;
        }

        if (wasInBs(info) && info.getStatusBsSynced() == StatusBsSynced.SENDING) {
            return true;
        }

        if (CampaignTypeKinds.WALLET.contains(info.getType())) {
            if (info.getStatusBsSynced() == StatusBsSynced.SENDING) {
                return true;
            } else {
                logger.debug("Strange state: wallet {} is not sending: {}", info.getId(), info.getStatusBsSynced());
            }
        }

        logger.trace("Filtered campaign {}: not eligible state or type", info.getId());
        return false;
    }

    // как $BS::Export::SQL_MONEY_COND из perl, но без условия на BannerID
    // (потому что мы не джойним баннеры к кампаниям)
    private static boolean moneyCondition(QueuedCampaign info) {
        return info.getHasMoney()
                || !info.getStatusArchived() && wasInBs(info)
                || !info.getStatusArchived() && CampaignTypeKinds.INTERNAL.contains(info.getType());
    }

    private static boolean wasInBs(QueuedCampaign info) {
        return info.getOrderId() != 0;
    }
}
