package ru.yandex.direct.bsexport.snapshot;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import org.jooq.DSLContext;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.bsexport.snapshot.model.ExportedCampaign;
import ru.yandex.direct.bsexport.snapshot.model.ExportedClient;
import ru.yandex.direct.bsexport.snapshot.model.ExportedUser;
import ru.yandex.direct.bsexport.snapshot.model.QueuedCampaign;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.core.entity.client.model.CurrencyConvertType;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;

import static ru.yandex.direct.core.entity.campaign.repository.CampaignRepository.WALLETS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENT_CURRENCY_CHANGES;
import static ru.yandex.direct.dbschema.ppc.Tables.CURRENCY_CONVERT_QUEUE;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.dbschema.ppc.tables.BsExportQueue.BS_EXPORT_QUEUE;
import static ru.yandex.direct.dbschema.ppc.tables.CampOptions.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbutil.SqlUtils.STRAIGHT_JOIN;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromFields;


@Repository
public class SnapshotRepository {
    private final JooqReaderWithSupplier<QueuedCampaign> queuedCampaignMapper;
    private final JooqReaderWithSupplier<ExportedCampaign> exportedCampaignMapper;
    private final JooqReaderWithSupplier<ExportedUser> exportedUserMapper;
    private final JooqReaderWithSupplier<ExportedClient> exportedClientMapper;

    public SnapshotRepository() {
        queuedCampaignMapper = createQueuedCampaignMapper();
        exportedCampaignMapper = createExportedCampaignMapper();
        exportedUserMapper = createExportedUserMapper();
        exportedClientMapper = createExportedClientMapper();
    }

    public Map<Long, QueuedCampaign> getQueuedCampaigns(DSLContext dslContext,
                                                 Collection<Long> campaignIds,
                                                 int limit) {
        // TODO DIRECT-112378: для ре-экспорта выбирать кампании из очереди с сортировкой по другим колонкам
        return dslContext.select(queuedCampaignMapper.getFieldsToRead())
                .hint(STRAIGHT_JOIN)
                .from(BS_EXPORT_QUEUE)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BS_EXPORT_QUEUE.CID))
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .leftJoin(WALLETS).on(WALLETS.CID.eq(CAMPAIGNS.WALLET_CID))
                .where(BS_EXPORT_QUEUE.CID.in(campaignIds))
                .orderBy(BS_EXPORT_QUEUE.SEQ_TIME, BS_EXPORT_QUEUE.QUEUE_TIME, BS_EXPORT_QUEUE.CID)
                .limit(limit)
                .fetchMap(CAMPAIGNS.CID, queuedCampaignMapper::fromDb);
    }

    public Map<Long, ExportedCampaign> getExportedCampaigns(DSLContext dslContext, Collection<Long> campaignIds) {
        return dslContext.select(exportedCampaignMapper.getFieldsToRead())
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .fetchMap(CAMPAIGNS.CID, exportedCampaignMapper::fromDb);
    }

    public Map<Long, ExportedUser> getUsers(DSLContext dslContext, Collection<Long> userIds) {
        return dslContext.select(exportedUserMapper.getFieldsToRead())
                .from(USERS)
                .where(USERS.UID.in(userIds))
                .fetchMap(USERS.UID, exportedUserMapper::fromDb);
    }

    public Map<Long, ExportedClient> getClients(DSLContext dslContext, Collection<Long> clientIds) {
        return dslContext.select(exportedClientMapper.getFieldsToRead())
                .from(CLIENTS)
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .leftJoin(CLIENT_CURRENCY_CHANGES).on(CLIENT_CURRENCY_CHANGES.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .leftJoin(CURRENCY_CONVERT_QUEUE).on(CURRENCY_CONVERT_QUEUE.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .where(CLIENTS.CLIENT_ID.in(clientIds))
                .fetchMap(CLIENTS.CLIENT_ID, exportedClientMapper::fromDb);
    }

    private static JooqReaderWithSupplier<QueuedCampaign> createQueuedCampaignMapper() {
        return JooqReaderWithSupplierBuilder.builder(QueuedCampaign::new)
                .readProperty(QueuedCampaign.ID, fromField(CAMPAIGNS.CID))
                .readProperty(QueuedCampaign.ORDER_ID, fromField(CAMPAIGNS.ORDER_ID))
                .readProperty(QueuedCampaign.WALLET_ID, fromField(CAMPAIGNS.WALLET_CID))
                .readProperty(QueuedCampaign.STATUS_ARCHIVED,
                        fromField(CAMPAIGNS.ARCHIVED).by(CampaignMappings::archivedFromDb))
                .readProperty(QueuedCampaign.TYPE,
                        fromField(CAMPAIGNS.TYPE).by(CampaignType::fromSource))
                .readProperty(QueuedCampaign.STATUS_BS_SYNCED,
                        fromField(CAMPAIGNS.STATUS_BS_SYNCED).by(CampaignMappings::statusBsSyncedFromDb))
                .readProperty(QueuedCampaign.SEQUENCE_TIME, fromField(BS_EXPORT_QUEUE.SEQ_TIME))
                .readProperty(QueuedCampaign.QUEUE_TIME, fromField(BS_EXPORT_QUEUE.QUEUE_TIME))
                .readProperty(QueuedCampaign.STATUS_EMPTY,
                        fromField(CAMPAIGNS.STATUS_EMPTY).by(CampaignMappings::statusEmptyFromDb))
                .readProperty(QueuedCampaign.STATUS_MODERATE,
                        fromField(CAMPAIGNS.STATUS_MODERATE).by(CampaignStatusModerate::fromSource))
                .readProperty(QueuedCampaign.STATUS_POST_MODERATE,
                        fromField(CAMP_OPTIONS.STATUS_POST_MODERATE).by(CampaignStatusPostmoderate::fromSource))
                .readProperty(QueuedCampaign.HAS_MONEY,
                        fromFields(Set.of(CAMPAIGNS.SUM, CAMPAIGNS.SUM_SPENT, WALLETS.SUM, WALLETS.SUM_SPENT))
                                .by(BsExportMapping::campaignBalanceIsPositive))
                .build();
    }

    private static JooqReaderWithSupplier<ExportedUser> createExportedUserMapper() {
        return JooqReaderWithSupplierBuilder.builder(ExportedUser::new)
                .readProperty(ExportedUser.ID, fromField(USERS.UID))
                .readProperty(ExportedUser.STATUS_YANDEX_ADV,
                        fromField(USERS.STATUS_YANDEX_ADV).by(RepositoryUtils::booleanFromYesNo))
                .readProperty(ExportedUser.SHOW_ON_YANDEX_ONLY,
                        fromField(USERS.SHOW_ON_YANDEX_ONLY).by(RepositoryUtils::booleanFromYesNo))
                .build();
    }

    private static JooqReaderWithSupplier<ExportedClient> createExportedClientMapper() {
        return JooqReaderWithSupplierBuilder.builder(ExportedClient::new)
                .readProperty(ExportedClient.ID, fromField(CLIENTS.CLIENT_ID))
                .readProperty(ExportedClient.FAVICON_BLOCKED,
                        fromField(CLIENTS.IS_FAVICON_BLOCKED).by(RepositoryUtils::booleanFromLong))
                .readProperty(ExportedClient.HIDE_MARKET_RATING,
                        fromField(CLIENTS_OPTIONS.HIDE_MARKET_RATING).by(RepositoryUtils::booleanFromLong))
                .readProperty(ExportedClient.NON_RESIDENT,
                        fromField(CLIENTS_OPTIONS.NON_RESIDENT).by(RepositoryUtils::booleanFromLong))
                .readProperty(ExportedClient.IS_BUSINESS_UNIT,
                        fromField(CLIENTS_OPTIONS.IS_BUSINESS_UNIT).by(RepositoryUtils::booleanFromLong))
                .readProperty(ExportedClient.STATUS_BALANCE_BANNED,
                        fromField(CLIENTS_OPTIONS.STATUS_BALANCE_BANNED).by(RepositoryUtils::booleanFromYesNo))
                .readProperty(ExportedClient.OVERDRAFT_LIMIT, fromField(CLIENTS_OPTIONS.OVERDRAFT_LIM))
                .readProperty(ExportedClient.AUTO_OVERDRAFT_LIMIT, fromField(CLIENTS_OPTIONS.AUTO_OVERDRAFT_LIM))
                .readProperty(ExportedClient.DEBT, fromField(CLIENTS_OPTIONS.DEBT))
                .readProperty(ExportedClient.CURRENCY_CONVERT_TYPE,
                        fromField(CURRENCY_CONVERT_QUEUE.CONVERT_TYPE).by(CurrencyConvertType::fromSource))
                .readProperty(ExportedClient.CURRENCY_CONVERTED_TO,
                        fromField(CLIENT_CURRENCY_CHANGES.CURRENCY_TO).by(BsExportMapping::clientCurrencyChangeFromDb))
                .readProperty(ExportedClient.CURRENCY_CONVERT_DATE, fromField(CLIENT_CURRENCY_CHANGES.DATE))
                .readProperty(ExportedClient.FLAGS,
                        fromField(CLIENTS_OPTIONS.CLIENT_FLAGS).by(BsExportMapping.CLIENT_FLAGS_FROM_DB))
                .build();
    }

    private static JooqReaderWithSupplier<ExportedCampaign> createExportedCampaignMapper() {
        return JooqReaderWithSupplierBuilder.builder(ExportedCampaign::new)
                .readProperty(ExportedCampaign.ID, fromField(CAMPAIGNS.CID))
                .readProperty(ExportedCampaign.AUTO_OPTIMIZATION,
                        fromField(CAMPAIGNS.AUTO_OPTIMIZATION).by(RepositoryUtils::booleanFromYesNo))
                .build();
    }
}
