package ru.yandex.direct.bsexport.snapshot.holders;

import java.util.Collection;
import java.util.List;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;

import ru.yandex.direct.bsexport.snapshot.internal.HolderBase;
import ru.yandex.direct.core.entity.campaign.model.BillingAggregateCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.product.model.Product;
import ru.yandex.direct.core.entity.product.repository.ProductRepository;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;

@ParametersAreNonnullByDefault
public class BillingAggregatesHolder extends HolderBase<Long, BillingAggregateRequest, BillingAggregateCampaign> {
    private final CampaignTypedRepository campaignTypedRepository;
    private final ProductRepository productRepository;


    public BillingAggregatesHolder(
            CampaignTypedRepository campaignTypedRepository,
            ProductRepository productRepository,
            Supplier<Collection<Long>> idsSupplier) {
        super(idsSupplier);
        this.campaignTypedRepository = campaignTypedRepository;
        this.productRepository = productRepository;
    }

    @Override
    protected void firstFetchInternal(DSLContext dslContext, Collection<Long> clientIdsToFetch) {
        if (clientIdsToFetch.isEmpty()) {
            return;
        }
        var billingAggregateCampaignIds = dslContext
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.TYPE.eq(CampaignsType.billing_aggregate)
                        .and(CAMPAIGNS.CLIENT_ID.in(clientIdsToFetch)))
                .fetch(CAMPAIGNS.CID);
        if (billingAggregateCampaignIds.isEmpty()) {
            return;
        }
        List<BillingAggregateCampaign> billingAggregateCampaigns =
                campaignTypedRepository.getTypedCampaigns(dslContext, billingAggregateCampaignIds)
                        .stream()
                        .map(camp -> (BillingAggregateCampaign) camp)
                        .collect(Collectors.toList());
        var productIdToTypes = productRepository.getAllProducts()
                .stream()
                .collect(groupingBy(Product::getId, mapping(Product::getType, toList())));
        for (var billingAggregate : billingAggregateCampaigns) {
            var walletId = billingAggregate.getWalletId();
            var productTypes = productIdToTypes.get(billingAggregate.getProductId());
            checkState(!productTypes.isEmpty(), "Unknown productId {} for billing aggregate {}",
                    billingAggregate.getProductId(), billingAggregate.getId());
            for (var productType : productTypes) {
                put(new BillingAggregateRequest(walletId, productType), billingAggregate);
            }
        }

    }
}
