package ru.yandex.direct.bsexport.snapshot.holders;

import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bsexport.snapshot.QueuedCampaignsFilteringStep;
import ru.yandex.direct.bsexport.snapshot.SnapshotRepository;
import ru.yandex.direct.bsexport.snapshot.internal.HolderBase;
import ru.yandex.direct.bsexport.snapshot.model.QueuedCampaign;
import ru.yandex.direct.utils.CommonUtils;

@ParametersAreNonnullByDefault
public class QueuedCampaignsHolder extends HolderBase<Long, Long, QueuedCampaign> {
    private static final Logger logger = LoggerFactory.getLogger(QueuedCampaignsHolder.class);

    private final SnapshotRepository bsExportRepository;
    private final Supplier<Integer> dbRowsLimitSupplier;
    private final QueuedCampaignsFilteringStep filteringStep;

    public QueuedCampaignsHolder(SnapshotRepository bsExportRepository, QueuedCampaignsFilteringStep filteringStep,
                                 Supplier<Collection<Long>> campaignIdsSupplier,
                                 Supplier<Integer> dbRowsLimitSupplier) {
        super(campaignIdsSupplier);
        this.dbRowsLimitSupplier = dbRowsLimitSupplier;
        this.bsExportRepository = bsExportRepository;
        this.filteringStep = filteringStep;
    }

    public Set<Long> getCampaignAndWalletIds() {
        checkInitialized();

        HashSet<Long> result = new HashSet<>(getData().keySet());

        // добавляем id кампаний кошельков
        getData().values()
                .stream()
                .map(QueuedCampaign::getWalletId)
                .filter(CommonUtils::isValidId)
                .forEach(result::add);

        return result;
    }

    @Override
    protected void firstFetchInternal(DSLContext dslContext, Collection<Long> campaignIds) {
        int dbRowsLimit = dbRowsLimitSupplier.get();
        var campaigns = bsExportRepository.getQueuedCampaigns(dslContext, campaignIds, dbRowsLimit);
        logger.debug("Got {} campaigns from {} with {} db rows limit",
                campaigns.size(), campaignIds.size(), dbRowsLimit);

        filter(campaigns);
        putAll(campaigns);
    }

    private void filter(Map<Long, QueuedCampaign> campaigns) {
        var excessCampaignIds = filteringStep.collectExcessCampaignIds(campaigns.keySet());
        if (!excessCampaignIds.isEmpty()) {
            logger.info("Will be unlocked campaigns that exceeded db rows limit or not exists: {}", excessCampaignIds);
            excessCampaignIds.forEach(campaigns::remove);
        }

        var unsupportedCampaignIds = filteringStep.collectUnsupportedCampaignIds(campaigns.values());
        if (!unsupportedCampaignIds.isEmpty()) {
            logger.warn("Will be unlocked campaigns which types are not supported: {}", unsupportedCampaignIds);
            unsupportedCampaignIds.forEach(campaigns::remove);
        }

        var notEligibleCampaignIds = filteringStep.collectNotEligibleCampaignIds(campaigns.values());
        notEligibleCampaignIds.forEach(campaigns::remove);
    }
}
