package ru.yandex.direct.bsexport.snapshot.internal;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;

import static com.google.common.base.Preconditions.checkState;

/**
 * Обобщенный код для выборки, хранения и получения данных
 *
 * @param <S> тип исходных данных, по которым идет загрузка из базы
 * @param <K> тип ключа, по которому идет обращение к загруженным данным
 * @param <R> тип данных
 */
@ParametersAreNonnullByDefault
public abstract class HolderBase<S, K, R> implements FirstPassDataFetcher {
    private final Map<K, R> data;
    private final Supplier<Collection<S>> idsSupplier;

    private boolean initialized;

    protected HolderBase(Supplier<Collection<S>> idsSupplier) {
        this.idsSupplier = idsSupplier;

        data = new HashMap<>();
        initialized = false;
    }

    protected void checkInitialized() {
        checkState(initialized);
    }

    /**
     * Метод доступа к данным, проверяющий, что данные были загружены.
     * Для удаления данных из словаря - нужно использовать метод {@link #remove(Object)} Holder'а
     *
     * @return словарь данных, защищенный от модификации
     * @throws IllegalStateException если вызван до загрузки данных (через {@link #firstFetch(DSLContext)}
     */
    protected final Map<K, R> getData() {
        checkInitialized();
        return Collections.unmodifiableMap(data);
    }

    /**
     * Сохранить результат выборки данных в holder
     *
     * @param result данные для сохранения
     */
    protected final void putAll(Map<K, R> result) {
        data.putAll(result);
    }

    /**
     * Добавить один выбранный элемент в holder
     *
     * @param key   ключ по которому нужно сохранить данные
     * @param value значение для сохранения
     */
    protected final R put(K key, R value) {
        return data.put(key, value);
    }

    /**
     * Удалить один элемент из holder
     *
     * @param key ключ по которому нужно удалить данные
     */
    protected final R remove(K key) {
        return data.remove(key);
    }

    protected abstract void firstFetchInternal(DSLContext dslContext, Collection<S> idsToFetch);

    public final void firstFetch(DSLContext dslContext) {
        checkState(!initialized);

        Collection<S> ids = idsSupplier.get();
        firstFetchInternal(dslContext, ids);

        initialized = true;
    }

    /**
     * Получить данные по ключу
     *
     * @param key значение ключа по которому получаем данные
     * @throws IllegalStateException если вызван до загрузки данных (через {@link #firstFetch(DSLContext)}
     */
    public R getObject(K key) {
        checkInitialized();
        return data.get(key);
    }
}
