package ru.yandex.direct.canvas.tools_client;

import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.asynchttpclient.request.body.multipart.ByteArrayPart;
import org.asynchttpclient.request.body.multipart.PartBase;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.ErrorResponseWrapperException;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.canvas.CanvasToolsRequests.AdminRejectRequest;
import ru.yandex.direct.canvas.model.OnCreativeOperationResult;
import ru.yandex.direct.http.smart.annotations.Json;
import ru.yandex.direct.http.smart.annotations.ResponseHandler;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.http.smart.http.Body;
import ru.yandex.direct.http.smart.http.GET;
import ru.yandex.direct.http.smart.http.Headers;
import ru.yandex.direct.http.smart.http.Multipart;
import ru.yandex.direct.http.smart.http.POST;
import ru.yandex.direct.http.smart.http.Part;
import ru.yandex.direct.http.smart.http.Path;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;

import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;

public class CanvasToolsClient {
    private final Api api;

    private static final Logger logger = LoggerFactory.getLogger(CanvasToolsClient.class);

    public CanvasToolsClient(String url, TvmIntegration tvmIntegration, TvmService tvmService,
                             ParallelFetcherFactory parallelFetcherFactory) {
        api = createApi(url, tvmIntegration, tvmService, parallelFetcherFactory);
    }

    private Api createApi(String url, TvmIntegration tvmIntegration, TvmService tvmService,
                          ParallelFetcherFactory parallelFetcherFactory) {
        return Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withProfileName("canvas_tools_client")
                .useTvm(tvmIntegration, tvmService)
                .withBaseUrl(url)
                .build()
                .create(Api.class);
    }

    public String updateVideoVast(Long creativeId, String vastContent) {
        Result<String> result = api.updateVideoVast(creativeId, List.of(
                new ByteArrayPart("file",
                        vastContent.getBytes(StandardCharsets.UTF_8),
                        "multipart/form-data",
                        StandardCharsets.UTF_8,
                        creativeId + "-vast.xml"))).execute();
// check for 404
        List<Throwable> errors = result.getErrors();
        if (errors != null && errors.get(0) instanceof ErrorResponseWrapperException) {
            ErrorResponseWrapperException errorResponse = (ErrorResponseWrapperException) errors.get(0);
            if (errorResponse.getResponse() != null && errorResponse.getResponse().getStatusCode() == 404) {
                return "{\"error\": \"Creative not found\"}";
            }
        }
        // end check for 404

        checkResultForErrors(result, CanvasToolsClientException::new);
        return result.getSuccess();
    }

    public String getVideoVast(Long creativeId) {
        Result<String> result = api.getVideoVast(creativeId).execute();
// check for 404
        List<Throwable> errors = result.getErrors();
        if (errors != null && errors.get(0) instanceof ErrorResponseWrapperException) {
            ErrorResponseWrapperException errorResponse = (ErrorResponseWrapperException) errors.get(0);
            if (errorResponse.getResponse() != null && errorResponse.getResponse().getStatusCode() == 404) {
                return "{\"error\": \"Creative not found\"}";
            }
        }
        // end check for 404

        checkResultForErrors(result, CanvasToolsClientException::new);
        return result.getSuccess();
    }

    public String getCreativeData(Long creativeId) {
        Result<String> result = api.getCreativeData(creativeId).execute();

        // check for 404
        List<Throwable> errors = result.getErrors();
        if (errors != null && errors.get(0) instanceof ErrorResponseWrapperException) {
            ErrorResponseWrapperException errorResponse = (ErrorResponseWrapperException) errors.get(0);
            if (errorResponse.getResponse() != null && errorResponse.getResponse().getStatusCode() == 404) {
                return "{\"error\": \"Creative not found\"}";
            }
        }
        // end check for 404

        checkResultForErrors(result, CanvasToolsClientException::new);
        return result.getSuccess();
    }

    public String regenerateVideo(String videoId) {
        Result<String> result = api.regenerateVideo(videoId).execute();

        List<Throwable> errors = result.getErrors();
        if (errors != null) {
            for (Throwable error : errors) {
                logger.warn(error.getMessage());
            }
        }

        String taskId = result.getSuccess();
        if (taskId == null) {
            return result.getErrors().stream().map(Throwable::getMessage).collect(Collectors.joining(";"));
        }
        return taskId;
    }

    public String cmsRecode(String videoId) {
        Result<String> result = api.cmsRecode(videoId).execute();

        List<Throwable> errors = result.getErrors();
        if (errors != null) {
            for (Throwable error : errors) {
                logger.warn(error.getMessage());
            }
        }

        String taskId = result.getSuccess();
        if (taskId == null) {
            return result.getErrors().stream().map(Throwable::getMessage).collect(Collectors.joining(";"));
        }
        return taskId;
    }

    public String rebuildPackshotVideoConstructor(Set<String> fileIds) {
        Result<String> result = api.rebuildPackshotVideoConstructor(fileIds).execute();
        checkResultForErrors(result, CanvasToolsClientException::new);

        String taskId = result.getSuccess();
        if (taskId == null) {
            return result.getErrors().stream().map(Throwable::getMessage).collect(Collectors.joining(";"));
        }
        return taskId;
    }

    public String rebuildPreviewVideoConstructor(Set<String> fileIds) {
        Result<String> result = api.rebuildPreviewVideoConstructor(fileIds).execute();
        checkResultForErrors(result, CanvasToolsClientException::new);

        String taskId = result.getSuccess();
        if (taskId == null) {
            return result.getErrors().stream().map(Throwable::getMessage).collect(Collectors.joining(";"));
        }
        return taskId;
    }

    public Map<Long, OnCreativeOperationResult> sendToDirect(Set<Long> creativeIds) {
        return callOnIds(api.sendToDirect(creativeIds));
    }

    public Map<Long, OnCreativeOperationResult> sendToRtbHost(Set<Long> creativeIds) {
        return callOnIds(api.sendToRtbHost(creativeIds));
    }

    public Map<Long, OnCreativeOperationResult> reshootScreenshot(Set<Long> creativeIds) {
        return callOnIds(api.reshootScreenshot(creativeIds));
    }

    public Map<Long, OnCreativeOperationResult> rebuild(Set<Long> creativeIds) {
        return callOnIds(api.rebuild(creativeIds));
    }

    public Map<Long, OnCreativeOperationResult> adminReject(AdminRejectRequest request) {
        return callOnIds(api.adminRejectCreatives(request));
    }

    private Map<Long, OnCreativeOperationResult> callOnIds(Call<Map<Long, OnCreativeOperationResult>> call) {
        Result<Map<Long, OnCreativeOperationResult>> result = call.execute();
        checkResultForErrors(result, CanvasToolsClientException::new); // general http errors
        return result.getSuccess();
    }

    public interface Api {
        @POST("/tools/send_to_direct")
        @Json
        @Headers("Content-Type: application/json")
        Call<Map<Long, OnCreativeOperationResult>> sendToDirect(@Body Set<Long> creativeIds);

        @POST("/tools/send_to_rtbhost")
        @Json
        @Headers("Content-Type: application/json")
        Call<Map<Long, OnCreativeOperationResult>> sendToRtbHost(@Body Set<Long> creativeIds);

        @POST("/tools/reshoot_screenshot")
        @Json
        @Headers("Content-Type: application/json")
        Call<Map<Long, OnCreativeOperationResult>> reshootScreenshot(@Body Set<Long> creativeIds);

        @POST("/tools/rebuild")
        @Json
        @Headers("Content-Type: application/json")
        Call<Map<Long, OnCreativeOperationResult>> rebuild(@Body Set<Long> creativeIds);

        @GET("/tools/creative/{id}")
        @ResponseHandler
        Call<String> getCreativeData(@Path("id") Long creativeId);

        @GET("/tools/video-vast/{id}")
        @ResponseHandler
        Call<String> getVideoVast(@Path("id") Long creativeId);

        @POST("/tools/video-vast/{id}")
        @ResponseHandler
        @Multipart
        Call<String> updateVideoVast(@Path("id") Long creativeId, @Part List<PartBase> filesData);

        @POST("/tools/regenerate-video/{id}")
        @ResponseHandler
        @Headers("Content-Type: application/json")
        Call<String> regenerateVideo(@Path("id") String videoId);

        @POST("/tools/cms-recode/{id}")
        @ResponseHandler
        @Headers("Content-Type: application/json")
        Call<String> cmsRecode(@Path("id") String videoId);

        @POST("/tools/constructor-files/rebuild_packshot")
        @ResponseHandler
        @Headers("Content-Type: application/json")
        Call<String> rebuildPackshotVideoConstructor(@Body @Json Set<String> fileIds);

        @POST("/tools/constructor-files/rebuild_preview")
        @ResponseHandler
        @Headers("Content-Type: application/json")
        Call<String> rebuildPreviewVideoConstructor(@Body @Json Set<String> fileIds);

        @POST("/tools/admin_reject_creatives")
        @Json
        @Headers("Content-Type: application/json")
        Call<Map<Long, OnCreativeOperationResult>> adminRejectCreatives(@Body @Json AdminRejectRequest request);
    }
}
