package ru.yandex.direct.cloud.iam.service;

import java.io.Closeable;
import java.util.function.Consumer;

import org.springframework.scheduling.TaskScheduler;

import ru.yandex.direct.cloud.iam.IOauthTokenProvider;
import ru.yandex.direct.liveresource.LiveResource;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.liveresource.PollingLiveResourceWatcher;

/**
 * Следит за обновлениями OAUTH-токена в файле, и уведомляет подписчика, если токен в файле изменился
 */
class LiveResourceOAuthTokenProvider implements IOauthTokenProvider, Closeable {
    private static final int CHECK_OAUTH_TOKEN_TIME_INTERVAL_IN_MS = 5000;
    private String oAuthToken;
    private final Consumer<String> oauthTokenChangedConsumer;
    private final PollingLiveResourceWatcher oAuthTokenWatcher;

    public LiveResourceOAuthTokenProvider(
            String oAuthTokenUrl, TaskScheduler taskScheduler, Consumer<String> oauthTokenChangedConsumer) {
        this.oauthTokenChangedConsumer = oauthTokenChangedConsumer;
        LiveResource oauthTokenResource = LiveResourceFactory.get(oAuthTokenUrl);

        oAuthTokenWatcher = new PollingLiveResourceWatcher(
                oauthTokenResource, oauthTokenResource.getContent(), taskScheduler,
                CHECK_OAUTH_TOKEN_TIME_INTERVAL_IN_MS);

        oAuthToken = getOAuthTokenFromResourceContent(oauthTokenResource.getContent());

        oAuthTokenWatcher.addListener(e -> updateToken(e.getCurrentContent()));
        oAuthTokenWatcher.watch();
    }

    private void updateToken(String currentResourceContent) {
        currentResourceContent = getOAuthTokenFromResourceContent(currentResourceContent);
        if (!currentResourceContent.equals(oAuthToken)) {
            oAuthToken = currentResourceContent;
            this.oauthTokenChangedConsumer.accept(oAuthToken);
        }
    }

    private String getOAuthTokenFromResourceContent(String content) {
        if (content == null) {
            return "";
        }
        return content.trim();
    }

    @Override
    public String getCurrentOAuthToken() {
        return oAuthToken;
    }

    @Override
    public void close() {
        oAuthTokenWatcher.close();
    }
}
