package ru.yandex.direct.cloud.mdb.mysql.api.service;

import java.io.Closeable;
import java.io.IOException;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.cloud.iam.CloudApiConnectionInfo;
import ru.yandex.direct.cloud.iam.IIamTokenProvider;
import ru.yandex.direct.cloud.iam.service.CloudIamTokenProviderService;
import ru.yandex.direct.cloud.mdb.mysql.api.ICloudMdbMySqlApi;
import ru.yandex.direct.cloud.mdb.mysql.api.grpc.GrpcCloudMdbMySqlApiImpl;
import ru.yandex.direct.cloud.mdb.mysql.api.transport.GetClustersListRequest;
import ru.yandex.direct.cloud.mdb.mysql.api.transport.GetClustersListResponse;
import ru.yandex.direct.cloud.mdb.mysql.api.transport.GetLogRecordsListRequest;
import ru.yandex.direct.cloud.mdb.mysql.api.transport.GetLogRecordsListResponse;
import ru.yandex.direct.config.DirectConfig;

/**
 * Отдельный класс для создания сервисов для работы с api Яндекс.Облака для mySQL. Нужен, чтобы не можно было
 * привязать соломоновские метрики и трейсинг к вызовам апи (пока их тут нет, но потом будут), а также для возможности
 * создания нескольких сервисов для разных поставщиков iam-токенов, и для корректной очистки ресурсов
 * при завершении работы.
 */
@ParametersAreNonnullByDefault
public class CloudMdbMySqlApiService implements ICloudMdbMySqlApi, Closeable {
    private static final Logger logger = LoggerFactory.getLogger(CloudMdbMySqlApiService.class);

    private final GrpcCloudMdbMySqlApiImpl cloudMdbMySqlApi;

    /**
     * Создает сервис для работы с api Яндекс.Облака для кластеров mySQL
     * @param connectionInfo параметры соединения с api Яндекс.Облака
     * @param iamTokenProvider поставщик iam-токенов для Яндекс.Облака
     */
    private CloudMdbMySqlApiService(CloudApiConnectionInfo connectionInfo, IIamTokenProvider iamTokenProvider) {
        cloudMdbMySqlApi = new GrpcCloudMdbMySqlApiImpl(connectionInfo, iamTokenProvider);
    }

    @Override
    public GetLogRecordsListResponse getLogRecordsList(GetLogRecordsListRequest request)
            throws IOException, InterruptedException {
        return cloudMdbMySqlApi.getLogRecordsList(request);
    }

    @Override
    public GetClustersListResponse getClustersList(GetClustersListRequest request)
            throws IOException, InterruptedException {
        return cloudMdbMySqlApi.getClustersList(request);
    }

    @PreDestroy
    @Override
    public void close() {
        cloudMdbMySqlApi.close();
    }

    /**
     * Создает экземпляр сервиса для работы с api Яндекс.Облака для кластеров mySQL. Параметры соединения с облаком
     * берутся из конфига Директа. Этот метод можно использовать для создания синглетона сервиса для работы
     * с api Яндекс.Облака для кластеров mySQL внутри спринга.
     * @param cloudBranch ветка cloud.api конфига Директа
     * @param iamTokenProvider поставщик iam-токенов для Яндекс.Облака
     * @return Сервис для работы с api Яндекс.Облака для кластеров mySQL
     */
    public static CloudMdbMySqlApiService create(DirectConfig cloudBranch, IIamTokenProvider iamTokenProvider) {
        CloudApiConnectionInfo connectionInfo =
                CloudIamTokenProviderService.getConnectionInfoFromConfig(cloudBranch.getBranch("api"));
        return new CloudMdbMySqlApiService(connectionInfo, iamTokenProvider);
    }

}
