package ru.yandex.direct.codegen.core.entity.banner;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.squareup.javapoet.TypeName;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.jooq.TableField;

import ru.yandex.direct.model.generator.old.conf.EnumConf;
import ru.yandex.direct.model.generator.old.conf.ModelConfFactory;
import ru.yandex.direct.model.generator.old.conf.ModelInterfaceConf;
import ru.yandex.direct.model.generator.old.conf.UpperLevelModelConf;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class OneToOneFlatBannerDataCollector implements DataCollector<OneToOneFlatBannerPartModel> {

    private static final String PROJECT_ROOT = "/Users/aleran/arc/arcadia/direct/";
    private static final String MODEL_CONF_ROOT =
            PROJECT_ROOT + "libs-internal/core-model/src/main/model-conf/bannernew";

    private static final Set<Predicate<String>> IMMUTABLE_PROPS_DETECTORS =
            ImmutableSet.of(
                    "id"::equals,
                    "campaignId"::equals,
                    "adGroupId"::equals);

    private static final Predicate<String> IS_IMMUTABLE_PROP_PREDICATE =
            str -> StreamEx.of(IMMUTABLE_PROPS_DETECTORS).reduce(Predicate::or).get().test(str);

    private static final Set<Predicate<String>> SYSTEM_PROPS_DETECTORS =
            ImmutableSet.of(
                    "id"::equals,
                    "campaignId"::equals,
                    "adGroupId"::equals,
                    prop -> prop.endsWith("StatusModerate"));

    private static final Predicate<String> IS_SYSTEM_PROP_PREDICATE =
            str -> StreamEx.of(SYSTEM_PROPS_DETECTORS).reduce(Predicate::or).get().test(str);

    private static final Set<String> WRITE_ONLY_PROPS = ImmutableSet.of("id", "campaignId", "adGroupId");

    private final ModelConfFactory confFactory;
    private final Console console;

    public OneToOneFlatBannerDataCollector() {
        this.confFactory = new ModelConfFactory();
        this.console = new Console();
    }

    @Override
    public OneToOneFlatBannerPartModel collectData() {
        // собираем данные о модели и немного о бизнес-логике
        ModelInterfaceConf modelConf = getModelConf();
        String packageName = getPackageName();
        String interfaceName = modelConf.getName();

        List<ModelPropInfo> allProps = extractPropsFromConfig(modelConf);
        List<ModelPropInfo> clientProps = getClientProperties(allProps);
        ModelPropInfo mainClientProp = getMainClientProperty(clientProps);
        List<ModelPropInfo> mutableProps = getMutableProperties(allProps);

        // собираем данные о таблице
        TableInfo table = getTable();
        List<DbFieldInfo> dbFields = tableToDbFields(table);

        // определяем соответствие между полями модели и полями таблицы
        Map<ModelPropInfo, DbFieldInfo> propToDbFieldsMap = getPropToDbFieldsMap(allProps, dbFields);

        List<ModelPropWithDbFieldInfo> modelPropWithDbFields =
                getModelPropWithDbFields(propToDbFieldsMap);

        Set<ModelPropInfo> mutablePropsSet = new HashSet<>(mutableProps);
        List<DbFieldInfo> mutableFields = EntryStream.of(propToDbFieldsMap)
                .filterKeys(mutablePropsSet::contains)
                .values()
                .distinct()
                .toList();

        List<ImportInfo> imports = getImports(modelPropWithDbFields);

        return (OneToOneFlatBannerPartModel) new OneToOneFlatBannerPartModel()
                .withPackageName(packageName)
                .withInterfaceName(interfaceName)
                .withInterfaceVariable(interfaceNameToInterfaceVariable(interfaceName))

                .withAllProps(allProps)
                .withClientProps(clientProps)
                .withMainClientProp(mainClientProp)

                .withMutableProps(mutableProps)

                .withTable(table)
                .withPropsWithFields(modelPropWithDbFields)
                .withMutableFields(mutableFields)

                .withImports(imports);
    }

    private List<ImportInfo> getImports(List<ModelPropWithDbFieldInfo> modelPropWithDbFields) {
        return StreamEx.of(modelPropWithDbFields)
                .filter(propWithField -> propWithField.getConverterToDb() != null)
                .map(propWithField -> new ImportInfo(propWithField.getProp().getTypeName().toString()))
                .toList();
    }

    private List<ModelPropInfo> extractPropsFromConfig(ModelInterfaceConf modelConf) {
        String className = modelConf.getName();
        Set<String> enums = StreamEx.of(modelConf.getEnums())
                .filter(enumConf -> enumConf.getSourceFile() != null)
                .map(EnumConf::getFullName)
                .toSet();
        return StreamEx.of(modelConf.getAttrs())
                .map(attrConf -> {
                    boolean isAutogeneratedEnum = enums.contains(attrConf.getType().toString());
                    return new ModelPropInfo(className, attrConf.getType(), isAutogeneratedEnum, attrConf.getName());
                })
                .toList();
    }

    private ModelInterfaceConf getModelConf() {
        console.printDelimiter();
        console.print("Укажите название, предварительно созданного, conf-файла нового интерфейса (пример: \"i_banner_with_turbolanding.conf\"): ");
        String confFileName = console.read();

        UpperLevelModelConf conf;
        try {
            URL confUrl = new URL("file", "", MODEL_CONF_ROOT + "/" + confFileName);
            conf = confFactory.createModelConf(confUrl);
        } catch (MalformedURLException e) {
            throw new RuntimeException(e);
        }
        checkState(conf instanceof ModelInterfaceConf);
        return (ModelInterfaceConf) conf;
    }

    private String getPackageName() {
        console.printDelimiter();
        console.print("Укажите название новой директории в bannernew/type (пример: \"turbolanding\"): ");
        return console.read();
    }

    private String interfaceNameToInterfaceVariable(String interfaceName) {
        if (interfaceName.startsWith("New")) {
            interfaceName = StringUtils.removeStart(interfaceName, "New");
        }
        return interfaceName.substring(0, 1).toLowerCase() + interfaceName.substring(1);
    }

    private List<ModelPropInfo> getClientProperties(List<ModelPropInfo> allProps) {
        console.printDelimiter();
        console.println("Укажите список \"клиентских\" свойств нового интерфейса, то есть таких, которые клиент ");
        console.println("может редактировать напрямую (в противоположность \"системным\" свойствам, " +
                "таким как статус модерации).");
        console.println("");

        List<ModelPropInfo> clientPropsCandidates = StreamEx.of(allProps)
                .mapToEntry(ModelPropInfo::getLowerCamel)
                .removeValues(IS_SYSTEM_PROP_PREDICATE)
                .keys()
                .toList();

        printPropsWithIndexes(clientPropsCandidates);
        List<ModelPropInfo> clientProps = getSublistByInputIndexes(clientPropsCandidates);
        checkState(!clientProps.isEmpty());

        String clientPropsStr = StringUtils.join(mapList(clientProps, ModelPropInfo::getUpperUnderscore), ", ");
        console.println();
        console.println("Выбранные клиентские свойства: " + clientPropsStr);

        return clientProps;
    }

    private ModelPropInfo getMainClientProperty(List<ModelPropInfo> clientProps) {
        if (clientProps.size() == 1) {
            return clientProps.iterator().next();
        }

        console.printDelimiter();
        console.println("Укажите название неотъемлемого свойства новой части баннера,");
        console.println("отсутствие которого в баннере говорит об отсутствии этой части целиком.");
        console.println("Это знание требуется репозиторию для определения, в каком случае необходимо");
        console.println("удалить запись из смежной таблицы.");
        console.println();
        console.println("Это должно быть одно из клиентских полей:");
        printPropsWithIndexes(clientProps);
        ModelPropInfo mainClientProp = getItemByInputIndex(clientProps);

        console.println();
        console.println("Выбранное основное клиентское свойство: " + mainClientProp.getUpperUnderscore());
        return mainClientProp;
    }

    private List<ModelPropInfo> getMutableProperties(List<ModelPropInfo> allProps) {
        console.printDelimiter();
        console.println("Укажите список мутабельных свойств нового интерфейса, то есть таких, которые могут");
        console.println("изменяться при обновлении баннера (пример мутабельного свойства - статус модерации,");
        console.println("пример иммутабельного - id баннера).");

        List<ModelPropInfo> mutablePropsCandidates = StreamEx.of(allProps)
                .mapToEntry(ModelPropInfo::getLowerCamel)
                .removeValues(IS_IMMUTABLE_PROP_PREDICATE)
                .keys()
                .toList();

        printPropsWithIndexes(mutablePropsCandidates);

        List<ModelPropInfo> mutableProps = getSublistByInputIndexes(mutablePropsCandidates);
        checkState(!mutableProps.isEmpty());

        String mutablePropsStr = StringUtils.join(mapList(mutableProps, ModelPropInfo::getUpperUnderscore), ", ");
        console.println();
        console.println("Выбранные мутабельные свойства: " + mutablePropsStr);

        return mutableProps;
    }

    private TableInfo getTable() {
        console.printDelimiter();
        console.print("Укажите название связанной таблицы (пример: \"banner_turbolandings\"): ");
        String sourceName = console.read();
        return new TableInfo(sourceName);
    }

    private List<DbFieldInfo> tableToDbFields(TableInfo table) {
        try {
            return tableNameToDbFieldsInner(table);
        } catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    private List<DbFieldInfo> tableNameToDbFieldsInner(TableInfo table) throws Exception {
        Class tableClass = getClassByName(table.getJooqClassNameFull());
        Field tableInstanceField = tableClass.getDeclaredField(table.getJooqUpperUnderscore());

        Object tableInstance = tableInstanceField.get(null);
        return StreamEx.of(tableClass.getDeclaredFields())
                .filter(field -> !Modifier.isStatic(field.getModifiers()))
                .mapToEntry(field -> (TableField) get(field, tableInstance))
                .mapKeys(Field::getName)
                .mapKeyValue((upperCaseName, tableField) ->
                        new DbFieldInfo(tableField.getType(), tableField.getName(), upperCaseName))
                .toList();
    }

    private Object get(Field field, Object obj) {
        try {
            field.setAccessible(true);
            return field.get(obj);
        } catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    private Map<ModelPropInfo, DbFieldInfo> getPropToDbFieldsMap(List<ModelPropInfo> allProps,
                                                                 List<DbFieldInfo> dbFieldInfos) {
        Map<ModelPropInfo, DbFieldInfo> propToDbFieldInfosMap = getPropToDbFieldsMapAuto(allProps, dbFieldInfos);

        console.printDelimiter();
        if (propToDbFieldInfosMap.isEmpty()) {
            console.println("Автоматически не удалось установить соответствие между полями модели и полями таблицы.");
        } else {
            console.println("Автоматически найденные соответствия между полями модели и полями таблицы:");
            propToDbFieldInfosMap.forEach((prop, field) -> console.println("- %s.%s -> %s",
                    prop.getModelClassNameShort(), prop.getUpperUnderscore(), field.getSourceName()));

            console.println();
            console.print("Для указанных выше полей модели соответствия определены верно? (Y/n): ");
            String answer = console.read();
            checkArgument(answer.equals("Y") || answer.equals("n"), "Допустимы только ответы \"Y\" или \"n\"");

            if (answer.equals("n")) {
                propToDbFieldInfosMap.clear();
            }
        }

        console.printDelimiter();
        if (propToDbFieldInfosMap.size() < allProps.size()) {
            console.println("Сейчас вы сможете указать соответствующие поля таблицы для тех полей модели, " +
                    "для которых их не удалось определить автоматически.");
            console.println("Нетривиальные маппинги вы сможете пропустить, чтобы позже самостоятельно добавить " +
                    "их в сгенерированный код репозиторного саппорта.");

            console.println();
            console.println("Поля модели, для которых не удалось определить соответствующие поля таблицы:");
            List<ModelPropInfo> unconnectedProps = StreamEx.of(allProps)
                    .remove(propToDbFieldInfosMap::containsKey)
                    .toList();
            List<DbFieldInfo> unconnectedFields = StreamEx.of(dbFieldInfos)
                    .remove(field -> propToDbFieldInfosMap.values().contains(field))
                    .toList();
            unconnectedProps.forEach(prop -> console.println("- %s.%s",
                    prop.getModelClassNameShort(), prop.getUpperUnderscore()));

            for (ModelPropInfo unconnectedProp : unconnectedProps) {
                console.println();
                console.println("Какое из полей таблицы соответствует полю модели %s.%s?: ",
                        unconnectedProp.getModelClassNameShort(), unconnectedProp.getUpperUnderscore());
                EntryStream.of(unconnectedFields)
                        .forKeyValue((index, field) -> console.println("- %s: %s",
                                index, field.getSourceName()));
                console.print("Введите индекс или \"-\", чтобы пропустить: ");
                String indexStr = console.read();
                if (indexStr.trim().equals("-")) {
                    continue;
                }
                Integer index = Integer.valueOf(indexStr);
                propToDbFieldInfosMap.put(unconnectedProp, unconnectedFields.get(index));
                unconnectedFields.remove((int) index);
            }

            console.println();
            console.println("Итоговое соответствие полей модели и полей таблицы:");
            propToDbFieldInfosMap.forEach((prop, field) -> console.println("- %s.%s -> %s",
                    prop.getModelClassNameShort(), prop.getUpperUnderscore(), field.getSourceName()));

            if (propToDbFieldInfosMap.size() < allProps.size()) {
                console.println("Маппинг определен не для всех полей. При необходимости вы можете самостоятельно " +
                        "добавить маппинг полей модели на поля таблицы в сгенерированный код репозиторного саппорта.");
            }
        }
        return propToDbFieldInfosMap;
    }

    private Map<ModelPropInfo, DbFieldInfo> getPropToDbFieldsMapAuto(
            List<ModelPropInfo> allProps, List<DbFieldInfo> dbFieldInfos) {
        final Map<String, String> defaultSearchMap = ImmutableMap.of(
                "id", "bid",
                "adGroupId", "pid",
                "campaignId", "cid"
        );

        Map<ModelPropInfo, String> propToPotentialSearchPropsMap = StreamEx.of(allProps)
                .mapToEntry(prop -> defaultSearchMap.getOrDefault(prop.getLowerCamel(), prop.getLowerCamel()))
                .toMap();

        Map<String, DbFieldInfo> lowerCaseDbFieldInfoNameToDbFieldInfoMap =
                listToMap(dbFieldInfos, DbFieldInfo::getSourceName);

        return EntryStream.of(propToPotentialSearchPropsMap)
                .mapValues(lowerCaseDbFieldInfoNameToDbFieldInfoMap::get)
                .nonNullValues()
                .toMap();
    }

    private List<ModelPropWithDbFieldInfo> getModelPropWithDbFields(
            Map<ModelPropInfo, DbFieldInfo> propToDbFieldsMap) {
        return EntryStream.of(propToDbFieldsMap)
                .mapKeyValue((prop, field) -> {
                    boolean read = !WRITE_ONLY_PROPS.contains(prop.getLowerCamel());
                    boolean write = true;

                    ModelPropWithDbFieldConverter converterToDb = getConverterToDb(prop, field);
                    ModelPropWithDbFieldConverter converterFromDb = getConverterFromDb(prop, field);

                    return new ModelPropWithDbFieldInfo(prop, field, read, write, converterToDb, converterFromDb);
                })
                .toList();
    }

    private ModelPropWithDbFieldConverter getConverterToDb(ModelPropInfo prop, DbFieldInfo field) {
        TypeName typeName = prop.getTypeName();
        if (field.getType() == Long.class && typeName.toString().equals("java.lang.Long")) {
            return null;
        }
        if (field.getType() == String.class && typeName.toString().equals("java.lang.String")) {
            return null;
        }
        if (field.getType().isEnum() && prop.isAutogeneratedEnum()) {
            String converterCall = prop.getTypeShort() + "::toSource";
            return new ModelPropWithDbFieldConverter(converterCall, null);
        }

        String converterMethodName = prop.getLowerCamel() + "ToDb";
        String converterCall = prop.getModelClassNameShort() + "RepositoryTypeSupport::" + converterMethodName;

        return new ModelPropWithDbFieldConverter(converterCall, converterMethodName);
    }

    private ModelPropWithDbFieldConverter getConverterFromDb(ModelPropInfo prop, DbFieldInfo field) {
        TypeName typeName = prop.getTypeName();
        if (field.getType() == Long.class && typeName.toString().equals("java.lang.Long")) {
            return null;
        }
        if (field.getType() == String.class && typeName.toString().equals("java.lang.String")) {
            return null;
        }
        if (field.getType().isEnum() && prop.isAutogeneratedEnum()) {
            String converterCall = prop.getTypeShort() + "::fromSource";
            return new ModelPropWithDbFieldConverter(converterCall, null);
        }

        String converterMethodName = prop.getLowerCamel() + "FromDb";
        String converterCall = prop.getModelClassNameShort() + "RepositoryTypeSupport::" + converterMethodName;

        return new ModelPropWithDbFieldConverter(converterCall, converterMethodName);
    }

    private Class getClassByName(String className) {
        try {
            return Class.forName(className);
        } catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    private void printPropsWithIndexes(List<ModelPropInfo> props) {
        EntryStream.of(props)
                .forKeyValue((index, prop) ->
                        console.println("  %s: %s.%s", index, prop.getModelClassNameShort(), prop.getUpperUnderscore()));
    }

    private <T> List<T> getSublistByInputIndexes(List<T> list) {
        console.println();
        console.print("Введите индексы, разделенные пробелами: ");

        String indexesStr = console.read();
        Set<Integer> indexes = StreamEx.of(indexesStr.split("\\s"))
                .map(String::trim)
                .remove(String::isEmpty)
                .map(Integer::valueOf)
                .toSet();
        return EntryStream.of(list)
                .filterKeys(indexes::contains)
                .values()
                .toList();
    }

    private <T> T getItemByInputIndex(List<T> list) {
        console.println();
        console.print("Введите индекс: ");

        String indexStr = console.read();
        return list.get(Integer.valueOf(indexStr.trim()));
    }
}
