package ru.yandex.direct.config;

import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalDouble;
import java.util.OptionalInt;
import java.util.OptionalLong;

import javax.annotation.Nonnull;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigObject;
import com.typesafe.config.ConfigValue;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

public class DirectConfig {
    private final Config conf;

    @Nonnull
    private final String prefix;

    public DirectConfig(Config conf) {
        this(conf, "");
    }

    private DirectConfig(Config conf, String prefix) {
        this.conf = conf;
        this.prefix = checkNotNull(prefix);
    }

    public Object getObject(String key) {
        return conf.getObject(prefixed(key));
    }

    public Optional<Object> findObject(String key) {
        String prefixed = prefixed(key);
        return conf.hasPath(prefixed) ? Optional.of(conf.getObject(prefixed)) : Optional.empty();
    }

    public String getString(String key) {
        return conf.getString(prefixed(key));
    }

    public Optional<String> findString(String key) {
        String prefixed = prefixed(key);
        return conf.hasPath(prefixed) ? Optional.of(conf.getString(prefixed)) : Optional.empty();
    }

    public int getInt(String key) {
        return conf.getInt(prefixed(key));
    }

    public OptionalInt findInt(String key) {
        String prefixed = prefixed(key);
        return conf.hasPath(prefixed) ? OptionalInt.of(conf.getInt(prefixed)) : OptionalInt.empty();
    }

    public long getLong(String key) {
        return conf.getLong(prefixed(key));
    }

    public OptionalLong findLong(String key) {
        String prefixed = prefixed(key);
        return conf.hasPath(prefixed) ? OptionalLong.of(conf.getLong(prefixed)) : OptionalLong.empty();
    }

    public double getDouble(String key) {
        return conf.getDouble(prefixed(key));
    }

    public OptionalDouble findDouble(String key) {
        String prefixed = prefixed(key);
        return conf.hasPath(prefixed) ? OptionalDouble.of(conf.getDouble(prefixed)) : OptionalDouble.empty();
    }

    public Duration getDuration(String key) {
        return conf.getDuration(prefixed(key));
    }

    public Optional<Duration> findDuration(String key) {
        String prefixed = prefixed(key);
        return conf.hasPath(prefixed) ? Optional.of(conf.getDuration(prefixed)) : Optional.empty();
    }

    public DirectConfig getBranch(String prefix) {
        return new DirectConfig(conf, this.prefix + prefix + ".");
    }

    public Optional<DirectConfig> findBranch(String prefix) {
        String path = this.prefix + prefix;
        return conf.hasPath(path) ? Optional.of(getBranch(prefix)) : Optional.empty();
    }

    public boolean getBoolean(String key) {
        return conf.getBoolean(prefixed(key));
    }

    public Optional<Boolean> findBoolean(String key) {
        String prefixed = prefixed(key);
        return conf.hasPath(prefixed) ? Optional.of(conf.getBoolean(prefixed)) : Optional.empty();
    }

    public List<String> getStringList(String key) {
        return conf.getStringList(prefixed(key));
    }

    public List<Long> getLongList(String key) {
        return conf.getLongList(prefixed(key));
    }

    public List<Integer> getIntList(String key) {
        return conf.getIntList(prefixed(key));
    }

    public Optional<List<String>> findStringList(String key) {
        String path = prefixed(key);
        return conf.hasPath(path) ? Optional.of(conf.getStringList(path)) : Optional.empty();
    }

    public List<DirectConfig> getConfigList(String key) {
        return conf.getObjectList(prefixed(key)).stream()
                .map(ConfigObject::toConfig)
                .map(DirectConfig::new)
                .collect(toList());
    }

    /**
     * Получить все настройки с полными путями и их значения
     */
    public Collection<Map.Entry<String, ConfigValue>> entrySet() {
        return getConfig().entrySet();
    }

    /**
     * Получить все настройки с полными путями и их стрингифицированными значениями
     */
    public Map<String, String> asMap() {
        return entrySet().stream().collect(toMap(Map.Entry::getKey, e -> e.getValue().unwrapped().toString()));
    }

    /**
     * Возвращает Typesafe'ный {@link Config}. Удобно для передачи ветки конфигурации во внешнюю библиотеку.
     */
    public Config getConfig() {
        Config subConf;
        if (prefix.isEmpty()) {
            subConf = conf;
        } else {
            subConf = conf.getConfig(prefix.substring(0, prefix.length() - 1));
        }
        return subConf;
    }

    /**
     * Проверить, есть ли в конфиге значение по указанному пути
     */
    public boolean hasPath(String key) {
        return conf.hasPath(prefixed(key));
    }

    private String prefixed(String key) {
        if (prefix.isEmpty()) {
            return key;
        } else {
            return prefix + key;
        }
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("DirectConfig{");
        sb.append("prefix='").append(prefix).append('\'');
        sb.append(", conf=").append(conf);
        sb.append('}');
        return sb.toString();
    }
}
