package ru.yandex.direct.config;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigParseOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.env.Environment;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.utils.io.FileUtils;

@ParametersAreNonnullByDefault
public class DirectConfigFactory {
    private static final Logger logger = LoggerFactory.getLogger(DirectConfigFactory.class);
    private static final Object CURRENT_MONITOR = new Object();
    private static final String CONFIG_FILE_ENV = "DIRECT_CONFIG_FILE";
    private static final String CONFIG_FILE_SYS_PROP = "config.override_file";
    private static final String APP_CONFIG_TMPL = "app-%s.conf";
    private static final String COMMON_CONFIG_TMPL = "common-%s.conf";
    private static volatile DirectConfig currentConfig;

    private DirectConfigFactory() {
    }

    public static DirectConfig getConfig() {
        return getConfig(Environment.getCached());
    }

    public static DirectConfig getCachedConfig() {
        DirectConfig result = currentConfig;
        if (result == null) {
            synchronized (CURRENT_MONITOR) {
                if (currentConfig == null) {
                    currentConfig = getConfig();
                }
                result = currentConfig;
            }
        }
        return result;
    }

    /**
     * Получить конфиг с переопределением. Ключи из overridingConfig имеют наивысший приоритет
     */
    public static DirectConfig getConfig(EnvironmentType env, Config overridingConfig) {
        Config commonConfig = readConfigFromClasspath(COMMON_CONFIG_TMPL, env);
        Config appConfig = readConfigFromClasspath(APP_CONFIG_TMPL, env);

        // был сделан для binlogbroker-json
        Config fileConfig = Optional.ofNullable(System.getenv(CONFIG_FILE_ENV))
                .map(DirectConfigFactory::readConfigFromFile)
                .orElse(ConfigFactory.empty());

        // для хотфиксов
        var systemPropFileConfigPath = System.getProperty(CONFIG_FILE_SYS_PROP);
        var systemPropFileConfig = ConfigFactory.empty();
        if (systemPropFileConfigPath != null) {
            var parsedConfig = readConfigFromFileOrNull(systemPropFileConfigPath);
            if (parsedConfig != null) {
                systemPropFileConfig = parsedConfig;
                logger.info("Successfully loaded config from System Property -D{}={}",
                        CONFIG_FILE_SYS_PROP, systemPropFileConfigPath);
            } else {
                logger.warn("Failed to load config from System Property -D{}={}, File does not exist.",
                        CONFIG_FILE_SYS_PROP, systemPropFileConfigPath);
            }
        }

        // генерируется из System Properties
        Config systemPropsConfig = ConfigFactory.systemProperties();

        Config conf = overridingConfig
                .withFallback(systemPropsConfig)
                .withFallback(systemPropFileConfig)
                .withFallback(fileConfig)
                .withFallback(appConfig)
                .withFallback(commonConfig)
                .resolve();
        logger.debug("merged config for {}: {}", env, conf);
        return new DirectConfig(conf);
    }


    /**
     * Получить конфиг с переопределением. Ключи из overridingConfig имеют наивысший приоритет
     */
    public static DirectConfig getConfig(EnvironmentType env, Map<String, Object> overridingConfig) {
        return getConfig(env, ConfigFactory.parseMap(overridingConfig));
    }

    public static DirectConfig getConfig(EnvironmentType env) {
        return getConfig(env, Collections.emptyMap());
    }

    /**
     * Получить конфиг, прочитанный из файлов (без учёта System properties / environment)
     */
    public static DirectConfig getConfigWithoutSystem(EnvironmentType env) {
        Config commonConfig = readConfigFromClasspath(COMMON_CONFIG_TMPL, env);
        Config appConfig = readConfigFromClasspath(APP_CONFIG_TMPL, env);
        Config fileConfig = Optional.ofNullable(System.getenv(CONFIG_FILE_ENV))
                .map(DirectConfigFactory::readConfigFromFile)
                .orElse(ConfigFactory.empty());
        Config conf = fileConfig
                .withFallback(appConfig)
                .withFallback(commonConfig)
                .resolve();
        logger.debug("merged config for {}: {}", env, conf);
        return new DirectConfig(conf);
    }

    private static Config readConfigFromClasspath(String configNameTmpl, EnvironmentType env) {
        Optional<String> configName = env.getHierarchy().stream()
                .map(e -> String.format(configNameTmpl, e.toString().toLowerCase()))
                .filter(x -> DirectConfigFactory.class.getResource("/" + x) != null)
                .findFirst();

        if (configName.isPresent()) {
            Config config = ConfigFactory
                    .parseResourcesAnySyntax(configName.get(),
                            ConfigParseOptions.defaults().setAllowMissing(false));
            logger.debug("readed config {} for {}: {}", configName.get(), env, config);
            return config;
        } else {
            logger.debug("return empty config for {}", env);
            return ConfigFactory.empty();
        }
    }

    static Config readConfigFromFile(String filePath) {
        return ConfigFactory.parseFileAnySyntax(FileUtils.expandHome(filePath).toFile(),
                ConfigParseOptions.defaults().setAllowMissing(false));
    }

    @Nullable
    static Config readConfigFromFileOrNull(String filePath) {
        var file = FileUtils.expandHome(filePath).toFile();
        if (!file.exists() || !file.isFile()) {
            return null;
        }
        return ConfigFactory.parseFileAnySyntax(file, ConfigParseOptions.defaults().setAllowMissing(false));
    }
}
