package ru.yandex.direct.config;

import javax.annotation.Nullable;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.support.PropertySourcesPlaceholderConfigurer;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MutablePropertySources;
import org.springframework.core.env.PropertySource;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;

import ru.yandex.direct.env.Environment;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.env.EnvironmentTypeProvider;

/**
 * Инициализация основных конфигурационных бинов
 */
@Configuration
@ComponentScan(
        basePackages = {"ru.yandex.direct.env", "ru.yandex.direct.liveresource"},
        excludeFilters = @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION)
)
public class EssentialConfiguration {
    public static final String ENVIRONMENT_TYPE = "environmentType";
    public static final String ENVIRONMENT_TYPE_PROVIDER_BEAN_NAME = "environmentTypeProvider";
    public static final String OVERRIDING_CONFIG_BEAN_NAME = "overridingConfig";
    public static final String CONFIG_SCHEDULER_BEAN_NAME = "liveConfigChangeTaskScheduler";

    @Bean(name = ENVIRONMENT_TYPE)
    static EnvironmentType environmentType() {
        return Environment.getCached();
    }

    /**
     * Определитель текущего типа окружения. Используем спринговый бин.
     */
    @Bean(name = ENVIRONMENT_TYPE_PROVIDER_BEAN_NAME)
    public EnvironmentTypeProvider environmentTypeProvider(EnvironmentType environmentType) {
        return () -> environmentType;
    }

    /**
     * С помощью параметра overridingConfig можно из приложения добавлять в конфиг дополнительные данные, например,
     * полученные из командной строки.
     */
    @Bean
    public static DirectConfig config(
            EnvironmentType env,
            @Nullable @Qualifier(OVERRIDING_CONFIG_BEAN_NAME) Config overridingConfig
    ) {
        if (overridingConfig == null) {
            overridingConfig = ConfigFactory.empty();
        }
        return DirectConfigFactory.getConfig(env, overridingConfig);
    }

    @Bean
    // @Autowired
    // метод static, чтобы spring мог создать этот бин как можно раньше, раньше инстанса AppConfig
    // и следовательно раньше бинов, которые могут быть сюда заинджекчены
    static PropertySourcesPlaceholderConfigurer properties(ConfigurableEnvironment env, DirectConfig config) {
        PropertySource<DirectConfig> directConfigSource = new DirectConfigPropertySource("conf", config);

        MutablePropertySources propertySources = env.getPropertySources();
        propertySources.addLast(directConfigSource);

        PropertySourcesPlaceholderConfigurer pspc = new PropertySourcesPlaceholderConfigurer();
        pspc.setPropertySources(propertySources);

        return pspc;
    }

    @Bean(name = CONFIG_SCHEDULER_BEAN_NAME)
    TaskScheduler liveConfigChangeTaskScheduler() {
        return new ThreadPoolTaskScheduler();
    }
}
