package ru.yandex.direct.db.config;

import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.net.FastUrlBuilder;

public class DbConfig {
    private static final Logger logger = LoggerFactory.getLogger(DbConfig.class);

    /**
     * Кол-во попыток подключиться к master-у Redis-а по умолчанию
     */
    private static final int DEFAULT_REQUEST_RETRIES = 3;

    public enum Engine {
        MYSQL,
        CLICKHOUSE
    }

    private List<String> hosts;
    private int port;
    private String user;
    private String pass;
    private String db;
    private Integer weight;
    private Engine engine = Engine.MYSQL;
    //connect timeout, in seconds
    private double connectTimeout;
    private boolean compression;
    /**
     * Кол-во попыток подключения к master-у Redis-а
     */
    private int requestRetries;
    private String dbName;
    private Map<String, String> extraUsers;

    /**
     * mysql - игнорирует эту опцию.
     * clickhouse - использовать http или https.
     */
    private boolean ssl;

    /**
     * mysql - игнорирует эту опцию.
     * clickhouse - следует ли отключить проверку SSL-сертификата. При ssl=false игнорируется.
     */
    private boolean verify = true;

    /**
     * mysql - игнорирует эту опцию
     * clickhouse - передаётся как настройка max_block_size при запросах
     *
     * null - дефолт в терминах clickhouse (т.е. шлём в clickhouse max_block_size=null)
     * 1024 - дефолт для Директовых логов (с целью уменьшить потребление памяти в clickhouse Директа)
     */
    private Integer clickhouseMaxBlockSize = 1024;

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    /**
     * @return Каталог в СУБД
     */
    public String getDb() {
        return db;
    }

    public void setDb(String db) {
        this.db = db;
    }

    public int getPort() {
        return port;
    }

    public void setPort(int port) {
        this.port = port;
    }

    public String getUser() {
        return user;
    }

    public void setUser(String user) {
        this.user = user;
    }

    public List<String> getHosts() {
        return hosts;
    }

    public void setHosts(List<String> hosts) {
        this.hosts = hosts;
    }

    public String getPass() {
        return pass;
    }

    public void setPass(String pass) {
        this.pass = pass;
    }

    public Integer getWeight() {
        return weight;
    }

    public void setWeight(Integer weight) {
        this.weight = weight;
    }

    public Engine getEngine() {
        return engine;
    }

    public void setEngine(Engine engine) {
        this.engine = engine;
    }

    public boolean isSsl() {
        return ssl;
    }

    public void setSsl(boolean ssl) {
        this.ssl = ssl;
    }

    public boolean isVerify() {
        return verify;
    }

    public void setVerify(boolean verify) {
        this.verify = verify;
    }

    public Integer getClickhouseMaxBlockSize() {
        return clickhouseMaxBlockSize;
    }

    public void setClickhouseMaxBlockSize(Integer clickhouseMaxBlockSize) {
        this.clickhouseMaxBlockSize = clickhouseMaxBlockSize;
    }

    /**
     * @return - connect-timeout в секундах
     */
    public double getConnectTimeout() {
        return connectTimeout;
    }

    /**
     * @return - connect-timeout в указанных юнитах
     */
    public long getConnectTimeout(TimeUnit unit) {
        long nanos = (long) ((double) TimeUnit.SECONDS.toNanos(1) * connectTimeout);
        return unit.convert(nanos, TimeUnit.NANOSECONDS);
    }

    public void setConnectTimeout(double newTimeout) {
        connectTimeout = newTimeout;
    }

    /**
     * @return Число попыток отправки запросов.
     */
    public int getRequestRetries() {
        if (requestRetries <= 0) {
            requestRetries = DEFAULT_REQUEST_RETRIES;
            logger.error(
                    "Parameter request_retries <= 0. Check db-config.json. Used default value {}",
                    DEFAULT_REQUEST_RETRIES);
        }
        return requestRetries;
    }

    public void setRequestRetries(int requestRetries) {
        this.requestRetries = requestRetries;
    }

    /**
     * @return Путь к конфигурации внутри dbconfig, разделитель между ключами - двоеточие
     */
    public String getDbName() {
        return dbName;
    }

    public void setDbName(String dbName) {
        this.dbName = dbName;
    }

    /**
     * @return Ключ - имя пользователя, значение - пароль.
     */
    public Map<String, String> getExtraUsers() {
        return extraUsers;
    }

    /**
     * @param extraUsers Ключ - имя пользователя, значение - пароль.
     */
    public void setExtraUsers(Map<String, String> extraUsers) {
        this.extraUsers = ImmutableMap.copyOf(extraUsers);
    }

    public boolean isCompression() {
        return compression;
    }

    public void setCompression(boolean compression) {
        this.compression = compression;
    }

    public String getJdbcUrl() {
        FastUrlBuilder builder = new FastUrlBuilder("jdbc:" + engine.toString().toLowerCase() + "://"
                + hosts.get(0) + ':' + port + "/" + db);
        if (engine == Engine.MYSQL) {
            builder.addParam("useUnicode", "true")
                    .addParam("useSSL", "false")
                    .addParam("characterEncoding", "utf8")
                    .addParam("useLocalSessionState", "true");
            if (compression) {
                builder.addParam("useCompression", "true");
            }
        } else if (engine == Engine.CLICKHOUSE) {
            if (!StringUtils.isEmpty(user) && !user.equals("default")) {
                builder.addParam("user", user);
            }
            if (!StringUtils.isEmpty(pass)) {
                builder.addParam("password", pass);
            }
            if (ssl) {
                builder.addParam("ssl", "1");
                builder.addParam("sslmode", verify ? "strict" : "none");
            }
        }
        return builder.build();
    }
}
